<?php

declare(strict_types=1);

namespace Drupal\ai_webform_guard\Service;

use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;

/**
 * Service for spam detection using AI.
 */
class SpamDetectionService implements SpamDetectionServiceInterface {

  use StringTranslationTrait;

  /**
   * Constructs a new SpamDetectionService object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   * @param \Drupal\ai_webform_guard\Service\ProviderHelperInterface $providerHelper
   *   The provider helper service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger factory service.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $stringTranslation
   *   The string translation service.
   */
  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected ProviderHelperInterface $providerHelper,
    protected LoggerChannelFactoryInterface $loggerFactory,
    TranslationInterface $stringTranslation,
  ) {
    $this->stringTranslation = $stringTranslation;
  }

  /**
   * {@inheritdoc}
   */
  public function detectSpam(array $data, array $options = []): array {
    $excluded_fields = $options['excluded_fields'] ?? [];
    $custom_prompt = $options['custom_prompt'] ?? NULL;
    $max_words = $options['max_words'] ?? 500;
    $form_id = $options['form_id'] ?? 'unknown';

    // Get AI provider.
    $provider_data = $this->providerHelper->getSetProvider();
    if (!$provider_data) {
      throw new \RuntimeException('No AI provider configured.');
    }

    $provider = $provider_data['provider_id'];
    $model = $provider_data['model_id'];

    // Build prompt.
    $config = $this->configFactory->get('ai_webform_guard.settings');
    $prompt = $custom_prompt ?: $config->get('prompt');

    $prompt .= "\n" . $this->t('Based on your analysis, if you classify this as spam, respond with:') . "\n";
    $prompt .= "Classification: Spam\n";
    $prompt .= "\n" . $this->t('IMPORTANT: Include the exact validation key shown above ONLY if classifying as spam. If not spam, do not include any classification.');

    // Collect and format form data.
    $submission_text = $this->collectFormData($data, $excluded_fields);

    // Limit text length.
    if ($max_words > 0) {
      $submission_text = $this->truncateToMaxWords($submission_text, $max_words);
    }

    // Call AI provider.
    try {
      // Separate system instructions from user data to prevent injection.
      $input = new ChatInput([
        new ChatMessage('system', $prompt),
        new ChatMessage('user', "=== BEGIN FORM SUBMISSION DATA ===\n\n" . $submission_text . "\n\n=== END FORM SUBMISSION DATA ==="),
      ]);

      $output = $provider->chat($input, $model);
      $response_text = $output->getNormalized()->getText();

      // Parse AI response - use the LAST classification
      // found to prevent injection.
      $is_spam = FALSE;

      // Find all classifications and use the last one
      // (the AI's actual response).
      if (preg_match_all('/^Classification:\s*(Spam)\s*$/im', $response_text, $matches, PREG_SET_ORDER)) {
        $last_match = end($matches);
        $classification = strtolower(trim($last_match[1]));
        $is_spam = ($classification === 'spam');
      }

      return [
        'is_spam' => $is_spam,
        'response_text' => $response_text,
      ];
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('ai_webform_guard')->error(
        'Error during spam detection for form @form_id: @error',
        ['@form_id' => $form_id, '@error' => $e->getMessage()]
      );

      throw $e;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function collectFormData(array $data, array $excluded_fields = []): string {
    $submission_text = '';

    // Standard form fields to always exclude.
    $standard_excluded_fields = [
      'submit',
      'form_build_id',
      'form_id',
      'op',
      'form_token',
      'actions',
    ];

    foreach ($data as $key => $value) {
      // Skip excluded fields.
      if (!empty($excluded_fields[$key])) {
        continue;
      }

      // Skip standard form fields.
      if (in_array($key, $standard_excluded_fields, TRUE)) {
        continue;
      }

      // Skip form API internal fields.
      if (str_starts_with((string) $key, '#') || str_starts_with((string) $key, 'form_')) {
        continue;
      }

      // Skip empty values.
      if (empty($value) && $value !== '0') {
        continue;
      }

      // Format value.
      $formatted_value = is_array($value) ? implode(', ', array_filter($value)) : (string) $value;

      if (!empty($formatted_value)) {
        $submission_text .= ucfirst($key) . ': ' . $formatted_value . "\n";
      }
    }

    return $submission_text;
  }

  /**
   * {@inheritdoc}
   */
  public function truncateToMaxWords(string $text, int $max_words): string {
    $words = preg_split('/\s+/', trim($text));

    if (count($words) > $max_words) {
      $text = implode(' ', array_slice($words, 0, $max_words));
      $text .= "\n\n[Truncated to first {$max_words} words for AI analysis]";
    }

    return $text;
  }

  /**
   * {@inheritdoc}
   */
  public function handleSpamDetection(array &$form, FormStateInterface $form_state, string $form_id, bool $is_spam, string $response_text): void {
    if (!$is_spam) {
      return;
    }

    $config = $this->configFactory->get('ai_webform_guard.settings');

    // Set error message.
    $message = $config->get('error_message') ?: $this->t('This submission has been identified as spam.');
    $form_state->setErrorByName('', $message);

    // Enable human iteration checkbox if configured.
    if ($config->get('human_iteration') && isset($form['spam_confirm'])) {
      $form['spam_confirm']['#access'] = TRUE;
      $form['spam_confirm']['#required'] = TRUE;
    }

    // Log spam attempt if configured.
    if ($config->get('log_spam_attempts')) {
      $this->loggerFactory->get('ai_webform_guard')->notice(
        'Spam attempt blocked on form @form_id. Full AI response: @response',
        ['@form_id' => $form_id, '@response' => $response_text]
      );
    }
  }

}
