<?php

namespace Drupal\ai_webform_guard\Form;

use Drupal\ai\Enum\AiModelCapability;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure AI Webform Guard module.
 */
class AIWebformGuardSettingsForm extends ConfigFormBase {

  use StringTranslationTrait;

  /**
   * Config settings.
   */
  const CONFIG_NAME = 'ai_webform_guard.settings';

  /**
   * Entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * AI Provider service.
   *
   * @var \Drupal\ai\AiProviderPluginManager
   */
  protected $providerManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->entityTypeManager = $container->get('entity_type.manager');
    $instance->providerManager = $container->get('ai.provider');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'ai_webform_guard_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      static::CONFIG_NAME,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {

    $form['prompt'] = [
      '#title' => $this->t('Webform guard generation prompt'),
      '#type' => 'textarea',
      '#description' => $this->t('Prompt used for generating the spam detection.'),
      '#required' => TRUE,
      '#config_target' => 'ai_webform_guard.settings:prompt',
    ];
    $form['longer_description'] = [
      '#theme' => 'item_list',
      '#items' => [
        $this->t('Prompt is rendered using Twig rendering engine and supports the following tokens:'),
        '{{ entity_lang_name }} - ' . $this->t('Human readable name of the entity language'),
      ],
    ];

    $models = $this->providerManager->getSimpleProviderModelOptions('chat', TRUE, TRUE, [
      AiModelCapability::ChatJsonOutput,
    ]);

    $form['ai_model'] = [
      '#title' => $this->t('AI provider/model'),
      '#type' => 'select',
      '#options' => $models,
      '#empty_option' => $this->t('Use Default Webform Spam Vision Model'),
      '#description' => $this->t('AI model to use for spam detection.'),
      '#config_target' => 'ai_webform_guard.settings:ai_model',
    ];

    $form['error_message'] = [
      '#title' => $this->t('Custom error message for spam submissions'),
      '#type' => 'textfield',
      '#description' => $this->t('This message will be shown if the submission is classified as spam. Leave empty to use the default message.'),
      '#config_target' => 'ai_webform_guard.settings:error_message',
    ];

    $form['log_spam_attempts'] = [
      '#title' => $this->t('Log spam attempts'),
      '#type' => 'checkbox',
      '#description' => $this->t('Check to log blocked spam submissions to the watchdog log.'),
      '#config_target' => 'ai_webform_guard.settings:log_spam_attempts',
    ];

    $form['human_iteration'] = [
      '#title' => $this->t('Human iteration'),
      '#type' => 'checkbox',
      '#description' => $this->t('Confirmation checkbox requiring human validation before allowing submission if flagged as spam.'),
      '#config_target' => 'ai_webform_guard.settings:human_iteration',
    ];

    $form['max_words'] = [
      '#title' => $this->t('Maximum number of words per AI request'),
      '#type' => 'number',
      '#min' => 30,
      '#description' => $this->t('Limits the number of words sent to the AI for analysis to reduce resource usage.
    On average, one word equals about 1.3–1.5 tokens. For example, 500 words ≈ 650–750 tokens.
    The fewer words you send, the fewer tokens (and energy/resources) are used.'),
      '#config_target' => 'ai_webform_guard.settings:max_words',
    ];

    $form['whitelist'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Allowed IP addresses'),
      '#description' => $this->t('Specify IP addresses that should bypass AI spam protection. Each IP address should be on a separate line. Wildcard (*) characters can be used (for example, 192.0.2.*).'),
      '#default_value' => implode("\n", $this->config(static::CONFIG_NAME)->get('whitelist') ?: []),
    ];

    $form['flood_control'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Flood control'),
      '#description' => $this->t('Log submissions marked as spam as flood attempt. When too many attempts are done, submissions are blocked without sending them to the AI. This protects your AI from being flooded with spam protection checks.'),
      '#config_target' => 'ai_webform_guard.settings:flood_control',
    ];

    $form['flood_window'] = [
      '#type' => 'number',
      '#title' => $this->t('Window'),
      '#min' => 1,
      '#description' => $this->t('The window of the flood control, in seconds.'),
      '#config_target' => 'ai_webform_guard.settings:flood_window',
      '#states' => [
        'visible' => [
          ':input[name="flood_control"]' => ['checked' => TRUE],
        ],
        'required' => [
          ':input[name="flood_control"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['flood_threshold'] = [
      '#type' => 'number',
      '#title' => $this->t('Threshold'),
      '#min' => 1,
      '#description' => $this->t('The threshold of the flood control.'),
      '#config_target' => 'ai_webform_guard.settings:flood_threshold',
      '#states' => [
        'visible' => [
          ':input[name="flood_control"]' => ['checked' => TRUE],
        ],
        'required' => [
          ':input[name="flood_control"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Process whitelist: convert textarea to array.
    $whitelist_raw = $form_state->getValue('whitelist');
    $whitelist = $whitelist_raw ? array_map('trim', explode("\n", $whitelist_raw)) : [];

    $this->config(static::CONFIG_NAME)
      ->set('prompt', $form_state->getValue('prompt'))
      ->set('ai_model', $form_state->getValue('ai_model'))
      ->set('error_message', $form_state->getValue('error_message'))
      ->set('log_spam_attempts', $form_state->getValue('log_spam_attempts'))
      ->set('human_iteration', $form_state->getValue('human_iteration'))
      ->set('max_words', $form_state->getValue('max_words'))
      ->set('whitelist', $whitelist)
      ->set('flood_control', $form_state->getValue('flood_control'))
      ->set('flood_window', $form_state->getValue('flood_window'))
      ->set('flood_threshold', $form_state->getValue('flood_threshold'))
      ->save();
    parent::submitForm($form, $form_state);
  }

}
