<?php

declare(strict_types=1);

namespace Drupal\ai_webform_guard\Hook;

use Drupal\ai_webform_guard\Service\ProviderHelperInterface;
use Drupal\ai_webform_guard\Service\SpamDetectionServiceInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\webform\WebformSubmissionInterface;

/**
 * Hook implementations for ai_webform_guard.
 */
final class AiWebformGuardFormHooks {
  use StringTranslationTrait;
  use DependencySerializationTrait;

  /**
   * Constructs a new AiWebformGuardFormHooks object.
   */
  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected ProviderHelperInterface $providerHelper,
    protected LoggerChannelFactoryInterface $loggerFactory,
    TranslationInterface $stringTranslation,
    protected SpamDetectionServiceInterface $spamDetectionService,
  ) {
    $this->stringTranslation = $stringTranslation;
  }

  /**
   * Implements hook_webform_submission_form_alter().
   */
  #[Hook('webform_submission_form_alter')]
  public function alterWebformSubmissionForm(array &$form, FormStateInterface $form_state, string $form_id): void {
    // Add this object as the validator - DependencySerializationTrait
    // will handle serialization.
    // Add a checkbox field to the form.
    $config = $this->configFactory->get('ai_webform_guard.settings');
    if ($config->get('human_iteration')) {
      $form['spam_confirm'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('I confirm that this submission is not spam.'),
        '#access' => FALSE,
        '#required' => FALSE,
        '#default_value' => $form_state->getUserInput()['spam_confirm'] ?? FALSE,
        '#weight' => -100,
      ];
    }

    $form['#validate'][] = [$this, 'validateSubmission'];
  }

  /**
   * Custom validation handler for webform submissions.
   */
  public function validateSubmission(array &$form, FormStateInterface $form_state): void {
    // Get the webform submission.
    $submission = $form_state->getFormObject()?->getEntity();

    // Skip validation if this is not a webform submission.
    if (!$submission instanceof WebformSubmissionInterface) {
      return;
    }

    // Check for any errors in the form state.
    if ($form_state->hasAnyErrors()) {
      return;
    }

    $user_input = $form_state->getUserInput();
    $spam_confirm = isset($user_input['spam_confirm']) && $user_input['spam_confirm'];

    // If spam_confirm is checked, skip spam classification validation.
    if ($spam_confirm) {
      return;
    }

    $webform_id = $submission->getWebform()->id();
    $config = $this->configFactory->get('ai_webform_guard.settings');
    $fields_config = $this->configFactory->get('ai_webform_guard.fields_settings')->get('settings_fields') ?? [];
    $excluded_fields = $fields_config[$webform_id]['excluded_fields'] ?? [];

    // Check if spam confirmation flow is enabled for this webform.
    $spam_flow_enabled = !empty($fields_config[$webform_id]['enable_spam_confirmation_flow']);

    // Get the prompt configuration.
    $custom_prompt = $fields_config[$webform_id]['custom_prompt'] ?? NULL;

    // Check if provider is configured.
    $provider_data = $this->providerHelper->getSetProvider();
    if (!$provider_data) {
      $form_state->setErrorByName('', $this->t('No AI provider configured.'));
      return;
    }

    // Get max_words from config.
    $max_words = (int) ($config->get('max_words') ?? 500);

    // Use SpamDetectionService to detect spam.
    try {
      $data = $submission->getData();
      $result = $this->spamDetectionService->detectSpam($data, [
        'excluded_fields' => $excluded_fields,
        'custom_prompt' => $custom_prompt,
        'max_words' => $max_words,
        'form_id' => $webform_id,
      ]);

      $is_spam = $result['is_spam'];
      $response_text = $result['response_text'];
      $flood_blocked = $result['flood_blocked'] ?? FALSE;

      // If submission was blocked by flood control, set error and return.
      if ($flood_blocked) {
        $form_state->setErrorByName('', $this->t('Too many spam attempts were logged by your IP. Please try again later.'));
        return;
      }

      // Handle email confirmation flow for webforms.
      $email_confirmation_is_configured = FALSE;

      if ($spam_flow_enabled) {
        $webform = $submission->getWebform();
        $handlers = $webform->getHandlers();

        foreach ($handlers as $handler) {
          $configuration = $handler->getConfiguration();
          if ($configuration['id'] === 'email_confirmation') {
            if ($is_spam) {
              // Enable the email_confirmation handler when spam is detected.
              $configuration['status'] = TRUE;
            }
            else {
              // Disable the email_confirmation handler
              // when no spam is detected.
              $configuration['status'] = FALSE;
            }
            $handler->setConfiguration($configuration);
            $webform->save();

            $email_confirmation_is_configured = TRUE;
          }
        }
      }

      // Handle spam detection only if email confirmation is not configured.
      if ($is_spam && !$email_confirmation_is_configured) {
        $this->spamDetectionService->handleSpamDetection($form, $form_state, $webform_id, $is_spam, $response_text);
      }
    }
    catch (\Exception $e) {
      // Error is already logged in SpamDetectionService.
      $form_state->setErrorByName('', $this->t('An error occurred during spam detection.'));
    }
  }

}
