<?php

declare(strict_types=1);

namespace Drupal\ai_webform_guard\Service;

use Drupal\Core\Form\FormStateInterface;

/**
 * Interface for spam detection service.
 */
interface SpamDetectionServiceInterface {

  /**
   * Detect if form submission is spam.
   *
   * @param array $data
   *   The form data to analyze.
   * @param array $options
   *   Options array containing:
   *   - excluded_fields: Array of field keys to exclude.
   *   - custom_prompt: Custom prompt to use (optional).
   *   - max_words: Maximum number of words to analyze.
   *   - form_id: The form ID for logging.
   *
   * @return array
   *   Array containing:
   *   - is_spam: Boolean indicating if submission is spam.
   *   - response_text: Full AI response text.
   *   - classification: The classification result (spam/not spam).
   */
  public function detectSpam(array $data, array $options = []): array;

  /**
   * Collect and format form data for AI analysis.
   *
   * @param array $data
   *   The form data.
   * @param array $excluded_fields
   *   Array of field keys to exclude.
   *
   * @return string
   *   The formatted submission text.
   */
  public function collectFormData(array $data, array $excluded_fields = []): string;

  /**
   * Truncate text to maximum number of words.
   *
   * @param string $text
   *   The text to truncate.
   * @param int $max_words
   *   Maximum number of words.
   *
   * @return string
   *   The truncated text.
   */
  public function truncateToMaxWords(string $text, int $max_words): string;

  /**
   * Handle spam detection result.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param string $form_id
   *   The form ID.
   * @param bool $is_spam
   *   Whether the submission is spam.
   * @param string $response_text
   *   The AI response text.
   */
  public function handleSpamDetection(array &$form, FormStateInterface $form_state, string $form_id, bool $is_spam, string $response_text): void;

}
