<?php

namespace Drupal\ajax_cart_update\Controller;

use Drupal\Core\Cache\CacheTagsInvalidatorInterface;
use Drupal\commerce_cart\CartProviderInterface;
use Drupal\commerce_order\Entity\Order;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides AJAX endpoints for cart updates.
 */
final class AjaxCartUpdateController implements ContainerInjectionInterface {
  use StringTranslationTrait;

  /**
   * Constructs a new AjaxCartUpdateController object.
   *
   * @param \Drupal\commerce_cart\CartProviderInterface $cart_provider
   *   The cart provider.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Cache\CacheTagsInvalidatorInterface $cache_tags_invalidator
   *   The cache tags invalidator.
   */
  public function __construct(
    protected CartProviderInterface $cart_provider,
    protected RendererInterface $renderer,
    protected LanguageManagerInterface $language_manager,
    protected LoggerChannelFactoryInterface $logger_factory,
    protected CacheTagsInvalidatorInterface $cache_tags_invalidator,
  ) {}

  /**
   * {@inheritdoc}
   *
   * @return static
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('commerce_cart.cart_provider'),
      $container->get('renderer'),
      $container->get('language_manager'),
      $container->get('logger.factory'),
      $container->get('cache_tags.invalidator')
    );
  }

  /**
   * Updates the cart and returns AJAX commands.
   */
  public function updateCart(Request $request): AjaxResponse {
    $response = new AjaxResponse();
    $form_data = $request->request->all();

    // Check if edit_quantity exists in the form.
    if (empty($form_data['edit_quantity'])) {
      return $response;
    }

    $carts = $this->cart_provider->getCarts();
    $updated = FALSE;
    // Handle the different ways edit_quantity might be structured.
    /** @var array<int|string, string> $quantities */
    $quantities = $form_data['edit_quantity'];

    // If quantities is a numerically indexed array, map it to order items.
    $is_numeric_indexed = array_keys($quantities) === range(0, count($quantities) - 1);

    foreach ($carts as $cart) {

      if ($cart instanceof Order && $cart->hasItems()) {
        $order_items = $cart->getItems();

        foreach ($order_items as $index => $order_item) {
          $item_id = $order_item->id();

          // Handle both associative and numeric arrays.
          $new_quantity = NULL;

          if ($is_numeric_indexed && isset($quantities[$index]) && is_numeric($quantities[$index])) {
            // If numeric indexed, use the array index.
            $new_quantity = (float) $quantities[$index];
          }
          elseif (isset($quantities[$item_id]) && is_numeric($quantities[$item_id])) {
            // If associative, use the item_id as key.
            $new_quantity = (float) $quantities[$item_id];
          }
          elseif (isset($quantities[(string) $index]) && is_numeric($quantities[(string) $index])) {
            // Handle associative array with numeric keys (e.g., "0", "1").
            $new_quantity = (float) $quantities[(string) $index];
          }

          if ($new_quantity !== NULL && $new_quantity >= 0) {
            $order_item->setQuantity((string) $new_quantity);
            try {
              $order_item->save();
              $updated = TRUE;
            }
            catch (\Exception $e) {
              $this->logger_factory->get('ajax_cart_update')->error('Failed to save order item @id: @error', [
                '@id' => $item_id,
                '@error' => $e->getMessage(),
              ]);
            }
          }
        }

        if ($updated) {
          try {
            $cart->save();
            // Clear cart cache.
            $this->cache_tags_invalidator->invalidateTags(['commerce_order:' . $cart->id()]);
          }
          catch (\Exception $e) {
            $this->logger_factory->get('ajax_cart_update')->error('Failed to save cart @id: @error.', [
              '@id' => $cart->id(),
              '@error' => $e->getMessage(),
            ]);
          }
        }
      }
    }

    // Return updated HTML for order summary.
    foreach ($carts as $cart) {
      if ($cart instanceof Order) {
        $order_total_summary = [
          '#theme' => 'commerce_order_total_summary',
          '#totals' => [
            'subtotal' => $cart->getSubtotalPrice() ? [
              'number' => $cart->getSubtotalPrice()->getNumber(),
              'currency_code' => $cart->getSubtotalPrice()->getCurrencyCode(),
            ] : [],
            'total' => $cart->getTotalPrice() ? [
              'number' => $cart->getTotalPrice()->getNumber(),
              'currency_code' => $cart->getTotalPrice()->getCurrencyCode(),
            ] : [],
          ],
        ];
        $rendered_summary = $this->renderer->renderInIsolation($order_total_summary);
        $response->addCommand(new ReplaceCommand('[data-drupal-selector="order-total-summary"]', $rendered_summary));

        // Update item prices.
        foreach ($cart->getItems() as $order_item) {
          $total_price = $order_item->getTotalPrice();

          if ($total_price) {
            $price_render = [
              '#type' => 'inline_template',
              '#template' => '{{ price|commerce_price_format }}',
              '#context' => ['price' => $total_price],
            ];
            $rendered_price = $this->renderer->renderInIsolation($price_render);
            $response->addCommand(new ReplaceCommand(".views-field-total-price__number[data-order-item-id='{$order_item->id()}']", $rendered_price));
          }
        }
        break;
      }
    }

    return $response;
  }

  /**
   * Returns cart summary HTML as JSON.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response containing cart summary HTML.
   */
  public function getCartSummaryHtml(): JsonResponse {
    $current_language = $this->language_manager->getCurrentLanguage()->getId();
    $this->language_manager->setConfigOverrideLanguage($this->language_manager->getLanguage($current_language));

    $carts = $this->cart_provider->getCarts();

    $response = [
      'order_total_summary' => '',
      'item_prices' => [],
      'items' => [],
      'total_quantity' => 0,
      'order_total' => '',
      'order_subtotal' => '',
    ];

    if (empty($carts)) {
      return new JsonResponse($response, 200, ['Content-Type' => 'application/json']);
    }

    foreach ($carts as $cart) {

      if ($cart instanceof Order) {
        $totals = [
          'subtotal' => [],
          'total' => [],
        ];

        $subtotal_price = $cart->getSubtotalPrice();
        $total_price = $cart->getTotalPrice();

        if ($subtotal_price) {
          $totals['subtotal'] = [
            'number' => $subtotal_price->getNumber(),
            'currency_code' => $subtotal_price->getCurrencyCode(),
          ];
          $subtotal_render = [
            '#type' => 'inline_template',
            '#template' => '{{ price|commerce_price_format }}',
            '#context' => ['price' => $subtotal_price],
          ];
          $response['order_subtotal'] = $this->renderer->renderInIsolation($subtotal_render);
        }

        if ($total_price) {
          $totals['total'] = [
            'number' => $total_price->getNumber(),
            'currency_code' => $total_price->getCurrencyCode(),
          ];
          $total_render = [
            '#type' => 'inline_template',
            '#template' => '{{ price|commerce_price_format }}',
            '#context' => ['price' => $total_price],
          ];
          $response['order_total'] = $this->renderer->renderInIsolation($total_render);
        }

        $order_total_summary = [
          '#theme' => 'commerce_order_total_summary',
          '#totals' => $totals,
        ];
        $rendered_summary = $this->renderer->renderInIsolation($order_total_summary);

        $wrapped_summary = [
          '#type' => 'container',
          '#attributes' => [
            'class' => [
              'field',
              'field--name-total-price',
              'field--type-commerce-price',
              'field--label-hidden',
              'field--item',
            ],
          ],
          'content' => [
            '#markup' => $rendered_summary,
          ],
        ];
        $response['order_total_summary'] = $this->renderer->renderInIsolation($wrapped_summary);

        if ($cart->hasItems()) {
          $total_quantity = 0;
          foreach ($cart->getItems() as $order_item) {
            $quantity = (float) $order_item->getQuantity();
            $total_quantity += $quantity;
            $total_price = $order_item->getTotalPrice();

            if ($total_price) {
              $price_render = [
                '#type' => 'inline_template',
                '#template' => '{{ price|commerce_price_format }}',
                '#context' => ['price' => $total_price],
              ];
              $response['item_prices'][] = $this->renderer->renderInIsolation($price_render);
            }
            else {
              $response['item_prices'][] = '';
            }
            $response['items'][] = [
              'order_item_id' => $order_item->id(),
              'quantity' => $order_item->getQuantity(),
              'title' => $order_item->getTitle(),
            ];
          }
          $response['total_quantity'] = $total_quantity;

        }
      }
    }

    return new JsonResponse($response, 200, ['Content-Type' => 'application/json']);
  }

  /**
   * Returns cart prices and related data as JSON.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response containing prices and related data.
   */
  public function getCartPricesHtml(): JsonResponse {
    $carts = $this->cart_provider->getCarts();
    $response = [
      'item_prices' => [],
      'total_price_number' => [],
      'cart_block_price' => [],
      'cart_block_summary_count' => '',
      'cart_block_quantity' => [],
      'cart_block_title' => [],
    ];

    foreach ($carts as $cart) {
      if ($cart instanceof Order && $cart->hasItems()) {
        $total_quantity = 0;
        foreach ($cart->getItems() as $order_item) {
          $quantity = (float) $order_item->getQuantity();
          $total_quantity += $quantity;
          $total_price = $order_item->getTotalPrice();

          // Price for item_prices, total_price_number, and cart_block_price.
          if ($total_price) {
            $price_render = [
              '#type' => 'inline_template',
              '#template' => '{{ price|commerce_price_format }}',
              '#context' => ['price' => $total_price],
            ];
            $rendered_price = $this->renderer->renderInIsolation($price_render);
            $response['item_prices'][] = $rendered_price;
            $response['total_price_number'][] = $rendered_price;
            $response['cart_block_price'][] = $rendered_price;
          }
          else {
            $response['item_prices'][] = '';
            $response['total_price_number'][] = '';
            $response['cart_block_price'][] = '';
          }

          // Add item quantities and titles for cart block.
          $response['cart_block_quantity'][] = $this->t('@quantity x', [
            '@quantity' => $order_item->getQuantity(),
          ], ['context' => 'Cart block quantity separator']);
          $response['cart_block_title'][] = $order_item->getTitle();
        }

        // Set total quantity for cart block summary.
        $response['cart_block_summary_count'] = $this->t('@count items', [
          '@count' => $total_quantity,
        ], ['context' => 'Cart block item count']);
      }
    }

    return new JsonResponse($response, 200, ['Content-Type' => 'application/json']);
  }

}
