<?php

namespace Drupal\ajax_cart_update\Hook;

use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Path\CurrentPathStack;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\views\ViewExecutable;

/**
 * Object-oriented implementations of hooks for ajax_cart_update.
 *
 * Class will be auto-registered as an autowired service and methods with the
 * #[Hook('...')] attribute will be run like normal hook implementations.
 */
class AjaxCartUpdateHooks {

  /**
   * The current path service.
   *
   * @var \Drupal\Core\Path\CurrentPathStack
   */
  protected CurrentPathStack $pathCurrent;
  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  public function __construct(CurrentPathStack $path_current, ConfigFactoryInterface $config_factory) {
    $this->pathCurrent = $path_current;
    $this->configFactory = $config_factory;
  }

  /**
   * Provides help text for the Ajax Cart Update module.
   *
   * @param string $route_name
   *   The name of the current route.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match object.
   *
   * @return string|null
   *   The help text HTML or NULL if not applicable.
   */
  #[Hook('help')]
  public function help($route_name, $route_match) {
    switch ($route_name) {
      case 'help.page.ajax_cart_update':
        $output = '';
        $output .= '<h3>' . t('About') . '</h3>';
        $output .= '<p>' . t('The Ajax Cart Update module provides an AJAX-based update for the Commerce cart form.') . '</p>';
        return $output;
    }
    return NULL;
  }

  /**
   * Alters the Commerce cart form to remove the submit button for AJAX updates.
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   * @param string $form_id
   *   The form ID.
   */
  #[Hook('form_alter')]
  public function formAlter(array &$form, FormStateInterface $form_state, string $form_id): void {
    if (strpos($form_id, 'views_form_commerce_cart_form') !== 0) {
      return;
    }

    // Ensure actions is an array before using it.
    if (empty($form['actions']) || !is_array($form['actions'])) {
      return;
    }

    // If submit exists, handle safely depending on its type.
    if (!array_key_exists('submit', $form['actions'])) {
      return;
    }

    $submit = $form['actions']['submit'];

    // If submit is an array check nested #name safely.
    if (is_array($submit)) {
      if (!empty($submit['#name']) && $submit['#name'] === 'edit-submit') {
        unset($form['actions']['submit']);
        return;
      }
      // If there are submit definitions, remove to enforce AJAX.
      unset($form['actions']['submit']);
      return;
    }

    // If submit is not an array (string/object/mixed), unset it safely.
    unset($form['actions']['submit']);
  }

  /**
   * Preprocesses variables for the page template.
   *
   * Attaches the appropriate Ajax Cart Update library based on
   * the update method when the current path matches the cart page.
   *
   * @param array<string, mixed> $variables
   *   An associative array containing the variables for the template.
   */
  #[Hook('preprocess_page')]
  public function preprocessPage(array &$variables): void {
    $current_path = $this->pathCurrent->getPath();
    $config = $this->configFactory->get('ajax_cart_update.settings');
    $update_method = $config->get('update_method') ?: 'selectors';

    if (preg_match('#^(/[a-z]{2})?/cart$#', $current_path)) {
      if (!isset($variables['#attached']['library'])) {
        $variables['#attached']['library'] = [];
      }
      $library = $update_method === 'endpoint' ? 'ajax_cart_update/ajax_cart_endpoint' : 'ajax_cart_update/ajax_cart_selectors';
      $variables['#attached']['library'][] = 'core/once';
      $variables['#attached']['library'][] = $library;
    }
  }

  /**
   * Preprocesses variables for the views_view template.
   *
   * Attaches selectors and update method settings for the Ajax Cart Update
   * module to drupalSettings for use in JavaScript.
   *
   * @param array $variables
   *   An associative array containing the variables for the template
   *   including a 'view' key with a \Drupal\views\ViewExecutable object.
   */
  #[Hook('preprocess_views_view')]
  public function preprocessViewsView(array &$variables): void {
    /** @var \Drupal\views\ViewExecutable|null $view */
    $view = $variables['view'] ?? NULL;
    if (!$view instanceof ViewExecutable) {
      return;
    }
    if (empty($variables['view']) || !is_object($variables['view'])) {
      return;
    }

    $view = $variables['view'];
    if ($view->id() !== 'commerce_cart_form' && $view->id() !== 'commerce_cart_block') {
      return;
    }

    $style_plugin = $view->style_plugin->getPluginId();
    $config = $this->configFactory->get('ajax_cart_update.settings');
    $update_method = $config->get('update_method') ?: 'selectors';

    // Define default selectors for the cart form.
    $selectors = [
      'total_price_number' => '.cart-form .views-field-total-price__number .field-content',
      'order_total_summary' => '[data-drupal-selector="order-total-summary"]',
      'order_total' => '.order-total-line__total .order-total-line-value',
      'order_subtotal' => '.order-total-line__subtotal .order-total-line-value',
      'cart_block_summary_count' => '.cart-block--summary__count',
      'cart_block_quantity' => '.cart--cart-block .views-field-quantity',
      'cart_block_price' => '.cart--cart-block .views-field-total-price__number .field-content',
      'cart_block_title' => '.cart--cart-block .views-field-title',
    ];

    // Define custom endpoints for endpoint method.
    $customEndpoints = [
      'summary' => [
        'url' => '/ajax/cart/summary-html',
        'selectors' => $selectors,
      ],
      'prices' => [
        'url' => '/ajax/cart/prices',
        'selectors' => [
          'total_price_number' => $selectors['total_price_number'],
          'cart_block_price' => $selectors['cart_block_price'],
          'cart_block_summary_count' => $selectors['cart_block_summary_count'],
          'cart_block_quantity' => $selectors['cart_block_quantity'],
          'cart_block_title' => $selectors['cart_block_title'],
        ],
      ],
    ];

    // Only process selectors if the update method is 'selectors'.
    if ($view->id() === 'commerce_cart_form') {
      switch ($style_plugin) {
        case 'table':
          // For table use Views default.
          $selectors['total_price_number'] = '.cart-form td.views-field-total-price__number';
          $selectors['cart_block_price'] = '.cart--cart-block .views-field-total-price__number';
          $customEndpoints['summary']['selectors'] = $selectors;
          $customEndpoints['prices']['selectors'] = [
            'total_price_number' => $selectors['total_price_number'],
            'cart_block_price' => $selectors['cart_block_price'],
            'cart_block_summary_count' => $selectors['cart_block_summary_count'],
            'cart_block_quantity' => $selectors['cart_block_quantity'],
            'cart_block_title' => $selectors['cart_block_title'],
          ];
          break;

        case 'grid':
          $selectors += [];
          break;

        case 'html_list':
          $selectors += [];
          break;
      }

    }
    elseif ($view->id() === 'commerce_cart_block') {
      // Pass update method even if selectors are not used.
      $variables['#attached']['drupalSettings']['ajaxCartUpdate'] = [
        'updateMethod' => $update_method,
      ];
    }

    if (!isset($variables['#attached']['drupalSettings']['ajaxCartUpdate'])) {
      $variables['#attached']['drupalSettings']['ajaxCartUpdate'] = [];
    }
    // Pass selectors and update method to JavaScript via drupalSettings.
    $key = $view->id() === 'commerce_cart_form' ? 'form' : 'block';
    $new_settings = [
      'selectors' => $selectors,
      'updateMethod' => $update_method,
      'customEndpoints' => $update_method === 'endpoint' ? $customEndpoints : [],
    ];

    // Merge with existing settings if any.
    $existing = $variables['#attached']['drupalSettings']['ajaxCartUpdate'] ?? [];
    $existing[$key] = array_replace_recursive($existing[$key] ?? [], $new_settings);
    $variables['#attached']['drupalSettings']['ajaxCartUpdate'] = $existing;

  }

}
