<?php

namespace Drupal\alert_message;

use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Render\Markup;
use Drupal\Core\Render\RendererInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a list controller for the alert message entity type.
 */
class AlertMessageListBuilder extends EntityListBuilder {

  /**
   * Constructs a new AlertMessageListBuilder object.
   */
  public function __construct(
    EntityTypeInterface $entity_type,
    EntityStorageInterface $storage,
    protected DateFormatterInterface $dateFormatter,
    protected RendererInterface $renderer,
  ) {
    parent::__construct($entity_type, $storage);
  }

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    return new static(
      $entity_type,
      $container->get('entity_type.manager')->getStorage($entity_type->id()),
      $container->get('date.formatter'),
      $container->get('renderer')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function render() {
    $build['table'] = parent::render();

    $total = $this->getStorage()
      ->getQuery()
      ->accessCheck(FALSE)
      ->count()
      ->execute();

    $build['summary']['#markup'] = $this->t('Total alert messages: @total', ['@total' => $total]);

    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function buildHeader() {
    $header['id'] = $this->t('ID');
    $header['label'] = $this->t('Label');
    $header['status'] = $this->t('Status');
    $header['uid'] = $this->t('Author');
    $header['users'] = $this->t('Users');
    $header['roles'] = $this->t('Roles');
    $header['publish_date'] = $this->t('Publish date');
    $header['unpublish_date'] = $this->t('Unpublish date');

    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity) {
    /** @var \Drupal\alert_message\Entity\AlertMessage $entity */
    $row['id'] = $entity->id();
    $row['label'] = $entity->toLink();
    $row['status'] = $entity->get('status')->value ? $this->t('Enabled') : $this->t('Disabled');
    $row['uid']['data'] = [
      '#theme' => 'username',
      '#account' => $entity->getOwner(),
    ];
    /** @var \Drupal\user\Entity\User $user */
    $users = array_map(function ($user) {
      $html = [
        '#theme' => 'username',
        '#account' => $user,
      ];
      return $this->renderer->render($html)->__toString();
    }, $entity->getTargetedUsers());
    $row['users'] = $users ? Markup::create(implode(', ', $users)) : $this->t('None');
    /** @var \Drupal\user\Entity\Role $role */
    $roles = array_map(function ($role) {
      return $role->label();
    }, $entity->getTargetedRoles()) ?: [$this->t('None')];
    $row['roles'] = implode(', ', $roles);
    $row['publish_date'] = $this->dateFormatter->format($entity->getPublishDate()->getTimestamp());
    $row['unpublish_date'] = $this->dateFormatter->format($entity->getUnpublishDate()->getTimestamp());

    return $row + parent::buildRow($entity);
  }

}
