<?php

namespace Drupal\alert_message\LazyBuilder;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Security\TrustedCallbackInterface;
use Drupal\Core\Session\AccountInterface;

/**
 * Lazy builder for the alert message block.
 */
class AlertMessageLazyBuilder implements TrustedCallbackInterface {

  /**
   * Constructs a new AlertMessageLazyBuilder object.
   *
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   The current user.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    protected AccountInterface $currentUser,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function trustedCallbacks() {
    return ['build'];
  }

  /**
   * Builds the alert message render array.
   *
   * @return array
   *   A renderable array.
   */
  public function build() {
    $alert_messages = $this->entityTypeManager->getStorage('alert_message')->loadByProperties([
      'status' => TRUE,
    ]);
    $alert_message_view_builder = $this->entityTypeManager->getViewBuilder('alert_message');

    /** @var \Drupal\alert_message\Entity\AlertMessage $alert_message */
    foreach ($alert_messages as $key => $alert_message) {
      // User doesn't have targeted role.
      if ($role_ids = $alert_message->getTargetedRoleIds()) {
        $user_roles = $this->currentUser->getRoles();
        if (!array_intersect($role_ids, $user_roles)) {
          unset($alert_messages[$key]);
          continue;
        }
      }
      // User is not in targeted user.
      if ($user_ids = $alert_message->getTargetedUserIds()) {
        if (!in_array($this->currentUser->id(), $user_ids)) {
          unset($alert_messages[$key]);
          continue;
        }
      }
      $alert_messages[$key] = $alert_message_view_builder->view($alert_message);
    }

    return [
      '#theme' => 'block__alert_messages',
      '#alert_messages' => $alert_messages,
      '#cache' => [
        'contexts' => ['cookies:alertMessageClosed', 'user', 'user.roles'],
        'tags' => ['alert_message_list'],
      ],
    ];
  }

}
