<?php

namespace Drupal\alt_login\Plugin\AltLoginMethod;

use Drupal\user\Entity\User;
use Drupal\alt_login\AltLoginMethodInterface;
use Drupal\user\UserInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation for logging in with the user name as an alias.
 *
 * @AltLoginMethod(
 *   id = "address_name",
 *   label = @Translation("Full name"),
 *   description = @Translation("the given name and family name from the address field")
 * )
 */
class AddressName implements AltLoginMethodInterface, ContainerFactoryPluginInterface {

  use StringTranslationTrait;

  /**
   * The name of the address field on the user entity.
   * @var string
   */
  private $fieldName;

  /**
   * @var Connection
   */
  private $database;

  /**
   * @var EntityFieldManagerInterface
   */
  private $entityFieldManager;

  /**
   * @param EmailValidator $entity_field_manager
   * @param Connection $database
   * @param MessengerInterface $messenger
   */
  function __construct(EntityFieldManagerInterface $entity_field_manager, Connection $database, MessengerInterface $messenger) {
    $this->entityFieldManager = $entity_field_manager;
    $this->database = $database;
    $this->messenger = $messenger;
  }

    /**
   *
   * @param ContainerInterface $container
   * @param array $configuration
   * @param string $plugin_id
   * @param array $plugin_definition
   * @return static
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static (
      $container->get('entity_field.manager'),
      $container->get('database'),
      $container->get('messenger')
    );
  }


  /**
   * {@inheritDoc}
   */
  function dedupeAlias(UserInterface $user) {
    $alias = $this->getAlias($user);
    if (empty($alias)) {
      $this->messenger->addWarning($this->t('Neither given name nor family name provided in address field.'));
    }
    $uids = $this->getUids($alias);
    if (!$user->isNew()) {
      unset($uids[array_search($user->id(), $uids)]);
    }
    return !empty($uids);
  }

  /**
   * {@inheritDoc}
   */
  function applies($alias) {
    // Pretty much any string could be valid
    return TRUE;
  }

  /**
   * {@inheritDoc}
   *
   * Its tricky to use entityQuery if the given and family names are not exactly
   * one word each.
   */
  function getUsernameFromAlias($alias) {
    if ($uids = $this->getUids($alias)) {
      return User::load(reset($uids))->getUsername();
    }
  }

  /**
   * {@inheritDoc}
   */
  function getAlias(UserInterface $user){
    $field_name = $this->fieldName();
    return implode(' ', [$user->{$field_name}->given_name, $user->{$field_name}->family_name]);
  }


  /**
   * Helper
   *
   * Get the name of the addressfield on the user.
   */
  private function fieldName() {
    if (empty($this->fieldName)) {
      foreach ($this->entityFieldManager->getFieldDefinitions('user', 'user') as $field_name => $fieldInfo) {
        if ($fieldInfo->getType() == 'address') {
          $this->fieldName  = $field_name;
          break;
        }
      }
    }
    return $this->fieldName;
  }

  private function getUids($alias) {
    $fname = $this->fieldName();
    $table = 'user__'.$fname;
    $query = $this->database->select($table, 'u')
      ->fields('u', ['entity_id'])
      ->where(
        "CONCAT_WS(:space, {$fname}_given_name, {$fname}_family_name) ='".$alias."'",
        [':space' => ' ']
      );
    return $query->execute()->fetchCol();
  }

}

