<?php

namespace Drupal\alt_text_validation\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBase;

/**
 * Defines the AltTextRule entity.
 *
 * @ConfigEntityType(
 *   id = "alt_text_rule",
 *   label = @Translation("Alt-Text Rule"),
 *   module = "alt_text_validation",
 *   handlers = {
 *     "list_builder" = "Drupal\alt_text_validation\AltTextRuleListBuilder",
 *      "form" = {
 *        "add" = "Drupal\alt_text_validation\Form\AltTextRuleForm",
 *        "edit" = "Drupal\alt_text_validation\Form\AltTextRuleForm",
 *        "default" = "Drupal\alt_text_validation\Form\AltTextRuleForm",
 *        "delete" = "Drupal\alt_text_validation\Form\AltTextRuleDeleteForm"
 *      },
 *      "route_provider" = {
 *        "html" = "Drupal\Core\Entity\Routing\AdminHtmlRouteProvider",
 *      }
 *   },
 *   config_prefix = "alt_text_rule",
 *   admin_permission = "administer alt text validation",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "rule_type" = "rule_type",
 *   },
 *   links = {
 *     "canonical" = "/admin/config/content/alt-text-validation/alt-text-rule/{alt_text_rule}",
 *     "add-form" = "/admin/config/content/alt-text-validation/alt-text-rule/add",
 *     "edit-form" = "/admin/config/content/alt-text-validation/alt-text-rule/{alt_text_rule}/edit",
 *     "delete-form" = "/admin/config/content/alt-text-validation/alt-text-rule/{alt_text_rule}/delete",
 *     "collection" = "/admin/config/content/alt-text-validation/alt-text-rule"
 *   },
 *   config_export = {
 *     "id",
 *     "label",
 *     "rule_type",
 *     "content_char_limit",
 *     "text_to_compare",
 *     "violation_action",
 *     "violation_message",
 *   }
 * )
 */
class AltTextRule extends ConfigEntityBase implements AltTextRuleInterface {

  /**
   * The Text Rule ID.
   *
   * @var string
   */
  protected string $id;

  /**
   * The Text Rule Label.
   *
   * @var string
   */
  protected string $label;

  /**
   * The rule type.
   *
   * @var string
   */
  protected $rule_type;

  /**
   * The content character limit.
   *
   * @var int
   */
  protected $content_char_limit;

  /**
   * The text to compare against.
   *
   * @var string
   */
  protected string $text_to_compare;

  /**
   * The action to take on violation.
   *
   * @var string
   */
  protected string $violation_action;

  /**
   * The message to display when an violation occurs.
   *
   * @var string
   */
  protected string $violation_message;

  /**
   * {@inheritdoc}
   */
  public function setRuleType(string $rule_type): AltTextRuleInterface {
    $this->rule_type = $rule_type;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getRuleType(): ?string {
    return $this->rule_type;
  }

  /**
   * {@inheritdoc}
   */
  public function getRuleTypeLabel() {
    $ruleType = $this->getRuleType();
    $options = $this->getRuleTypeOptions();
    return $options[$ruleType] ?? $ruleType;
  }

  /**
   * {@inheritdoc}
   */
  public function getRuleTypeOptions() {
    return self::RULE_TYPE_OPTIONS;
  }

  /**
   * {@inheritdoc}
   */
  public function setContentCharLimit(int $content_char_limit): AltTextRuleInterface {
    $this->content_char_limit = $content_char_limit;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getContentCharLimit(): ?int {
    return $this->content_char_limit;
  }

  /**
   * {@inheritdoc}
   */
  public function setTextToCompare(string $text_to_compare): AltTextRuleInterface {
    $this->text_to_compare = $text_to_compare;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getTextToCompare(): ?string {
    return $this->text_to_compare;
  }

  /**
   * {@inheritdoc}
   */
  public function setViolationAction(string $violation_action): AltTextRuleInterface {
    $this->violation_action = $violation_action;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getViolationAction(): ?string {
    return $this->violation_action;
  }

  /**
   * {@inheritdoc}
   */
  public function setViolationMessage(string $violation_message): AltTextRuleInterface {
    $this->violation_message = $violation_message;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getViolationMessage(): ?string {
    return $this->violation_message;
  }

  /**
   * {@inheritdoc}
   */
  public function isViolation(string $filename, string $alt, string $title): bool {
    // Determine if this rule is a violation based on the provided parameters.
    // Case race, first to true wins.
    $rule_type = $this->getRuleType();
    $text_to_compare = $this->getTextToCompare();
    switch (TRUE) {
      case (($rule_type === 'alt_is_filename') && ($alt === $filename)):
        return TRUE;

      case (($rule_type === 'alt_is_title') && ($alt === $title)):
        return TRUE;

      case ($rule_type === 'not_empty' && empty($alt)):
        return TRUE;

      case ($rule_type === 'length_limit' && strlen($alt) > (int) $this->getContentCharLimit()):
        return TRUE;

      case ($rule_type === 'not_begin_with' && stripos($alt, $text_to_compare) === 0):
        return TRUE;

      case (($rule_type === 'not_equal') && strcasecmp($alt, $text_to_compare) === 0):
        return TRUE;

      case ($rule_type === 'not_contain' && stripos($alt, $text_to_compare) !== FALSE):
        return TRUE;

      case ($rule_type === 'not_end_with' && strripos($alt, $text_to_compare) === (strlen($alt) - strlen($text_to_compare))):
        return TRUE;

      case ($rule_type === 'regex_match' && preg_match($text_to_compare, $alt)):
        return TRUE;

      default:
        return FALSE;
    }
  }

}
