<?php

namespace Drupal\alt_text_validation\Plugin\Validation\Constraint;

use Drupal\alt_text_validation\AtvCommonTools;
use Drupal\alt_text_validation\Service\ValidationToolsInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\ConstraintValidator;

// @todo inject the Validator service to process.

/**
 * Validates the UniqueInteger constraint.
 */
class AltTextRulesConstraintValidator extends ConstraintValidator implements ContainerInjectionInterface {

  /**
   * The validation tools service.
   *
   * @var \Drupal\alt_text_validation\Service\ValidationToolsInterface
   */
  protected $validationTools;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The configuration factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a AltTextRulesConstraintValidator object.
   *
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\alt_text_validation\Service\ValidationToolsInterface $validation_tools
   *   The validation tools service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory service.
   */
  public function __construct(MessengerInterface $messenger, ValidationToolsInterface $validation_tools, ConfigFactoryInterface $config_factory) {
    $this->messenger = $messenger;
    $this->validationTools = $validation_tools;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public function validate($value, Constraint $constraint) {
    // Get the configuration.
    $config = $this->configFactory->get('alt_text_validation.settings');
    $altTextValidationEnabled = $config->get('alt_text_validation_enabled') ?? 0;

    // Skip validation if alt_text_validation_enabled is off (0).
    if ($altTextValidationEnabled == 0) {
      return;
    }

    $validations = [];
    $is_image = AtvCommonTools::isImageField($value->getName(), $value->getEntity());
    foreach ($value as $item) {
      // @todo Determine whether it is a string field or an image field.
      if ($is_image) {
        $validations = $this->validationTools->validateImageField($value->getName(), $value->getEntity());
      }
      else {
        $validations = $this->validationTools->validateTextField($value->getName(), $value->getEntity());
      }
      if (!empty($validations)) {
        foreach ($validations as $filename => $validation) {
          if (!empty($validation['violations'])) {
            // Address the violations.
            // @todo Ideally bulleted list would be good.
            // @todo Fix message to indicate the image filename and field name.
            // @todo Fix the jump-link not showing up.
            $message = implode(', ', $validation['violations']);
            $this->context->buildViolation($message)
              ->setInvalidValue($validation['alt'])
              ->setPlural(1)
              // ->addMetaData('custom_data', ['error_code' => 'VIOLATION'])
              ->addViolation();
          }

          if (!empty($validation['warnings'])) {
            // Address the warnings.
            // @todo These need to be made more helpful.
            $this->messenger->addMessage(implode(', ', $validation['warnings']), 'warning');
          }
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function create($container) {
    return new static(
      $container->get('messenger'),
      $container->get('alt_text_validation.validationtools'),
      $container->get('config.factory')
    );
  }

}
