<?php

namespace Drupal\alt_text_validation\Plugin\views\area;

use Drupal\alt_text_validation\AtvCommonTools;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\State\State;
use Drupal\views\Attribute\ViewsArea;
use Drupal\views\Plugin\views\area\AreaPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines Views area plugin for the alt text validation summary.
 *
 * @ingroup views_area_handlers
 */
#[ViewsArea("alt_text_validation_report_summary")]
class AtvSummary extends AreaPluginBase {

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  public $dateFormatter;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\State
   */
  public $state;

  /**
   * Constructs a Block plugin.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   * @param Drupal\Core\State\State $state
   *   The state service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    DateFormatterInterface $date_formatter,
    State $state,
  ) {
    $this->areaType = 'header';
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->dateFormatter = $date_formatter;
    $this->state = $state;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('date.formatter'),
      $container->get('state')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function render($empty = FALSE) {
    $vars = [
      '@start' => $this->formatDate($this->state->get(AtvCommonTools::AUDIT_START_TIME_KEY, NULL)),
      '@end' => $this->formatDate($this->state->get(AtvCommonTools::AUDIT_END_TIME_KEY, NULL)),
      '@status' => $this->state->get(AtvCommonTools::AUDIT_STATUS_KEY, '?'),
    ];
    $status_text = $this->t('Audit status: @status', $vars);
    $start_text = $this->t('Started: @start', $vars);
    $end_text = $this->t('Completed: @end', $vars);
    $button_text = $this->t('Run a new audit');
    $output = [];
    $output['details'] = [
      '#prefix' => '<div class="atv-status">',
      '#markup' => "$status_text</br>$start_text</br>$end_text",
      '#suffix' => '</div>',
    ];
    $output['rebuild'] = [
      '#prefix' => '<div class="atv-rebuild"><a href="/admin/reports/alt-text-report/rebuild" class="button">',
      '#markup' => $this->t('Rebuild the audit'),
      '#suffix' => '</a></div>',
    ];
    return $output;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheMaxAge() {
    return 0;
  }

  /**
   * Optionally converts a timestamp to a date and time.
   *
   * @param int|null $timestamp
   *   The timestamp to convert.
   *
   * @return string
   *   The formatted timestamp or empty string if timestamp was empty.
   */
  protected function formatDate(?int $timestamp): string {
    if (!empty($timestamp)) {
      return $this->dateFormatter->format($timestamp, 'custom', 'm-d-Y h:i:s a');
    }
    return '';
  }

}
