<?php

namespace Drupal\alt_text_validation\Service;

use Drupal\alt_text_validation\AtvCommonTools;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class for Validator to validate image alt, file and title.
 */
class ValidationTools implements ValidationToolsInterface, ContainerInjectionInterface {

  use StringTranslationTrait;

  /**
   * The configuration for alt_text_validation.
   *
   * @var \Drupal\Core\Config\Config
   */
  protected $atvConfig;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;


  /**
   * The alt_text_validation logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Array of rule entities.
   *
   * @var \Drupal\alt_text_validation\Entity\AltTextRuleInterface[]
   */
  protected $rules = [];

  /**
   * Array of rule entities that are strict rules resulting in violations.
   *
   * @var \Drupal\alt_text_validation\Entity\AltTextRuleInterface[]?
   */
  protected $preventionRules = NULL;

  /**
   * Array of rule entities that are warning rules resulting in warnings.
   *
   * @var \Drupal\alt_text_validation\Entity\AltTextRuleInterface[]?
   */
  protected $warningRules = NULL;

  /**
   * Constructs the Validator service.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param Psr\Log\LoggerInterface $logger
   *   The alt_text_validation logger.
   */
  final public function __construct(
    ConfigFactoryInterface $config_factory,
    EntityTypeManagerInterface $entity_type_manager,
    LoggerInterface $logger,
  ) {
    $this->logger = $logger;
    $this->atvConfig = $config_factory->getEditable('alt_text_validation.settings');
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('entity_type.manager'),
      $container->get('logger.channel.alt_text_validation')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getViolations(string $filename, string $alt, string $title): array {
    $violation_messages = [];
    foreach ($this->getPreventionRules() as $rule) {
      if ($rule->isViolation($filename, $alt, $title)) {
        $violation_messages[$rule->label()] = $rule->getViolationMessage();
      }
    }
    return $violation_messages;
  }

  /**
   * {@inheritdoc}
   */
  public function getWarnings(string $filename, string $alt, string $title): array {
    $warning_messages = [];
    foreach ($this->getWarningRules() as $rule) {
      if ($rule->isViolation($filename, $alt, $title)) {
        $warning_messages[$rule->label()] = $rule->getViolationMessage();
      }
    }
    return $warning_messages;
  }

  /**
   * {@inheritdoc}
   */
  public function validateImageField(string $field_name, EntityInterface $entity): array {
    $validations = [];
    if (AtvCommonTools::isImageField($field_name, $entity)) {
      $field_values = $entity->get($field_name)->getValue();
      foreach ($field_values as $key => $field_value) {
        $file = $entity->get($field_name)?->referencedEntities()[$key];
        $file_uri = $file?->getFileUri();
        $file_source = $file->createFileUrl(FALSE);
        $apex_entity = $this->getApexEntity($entity);
        $validations[$file_source] = [
          'apex_entity_title' => $apex_entity->label(),
          'apex_entity_url' => ($apex_entity->isNew()) ? '' : $apex_entity->toUrl('canonical', ['absolute' => TRUE])->toString(),
          'field_name' => $field_name,
          'file_name' => $file_name = pathinfo($file_uri, PATHINFO_BASENAME),
          'file_source' => $file_source,
          'title' => $title = $field_value['title'] ?? '',
          'alt' => $alt = $field_value['alt'] ?? '',
          'violations' => $this->getViolations($file_name, $alt, $title),
          'warnings' => $this->getWarnings($file_name, $alt, $title),
          'entity_type' => $entity->getEntityTypeId(),
          'bundle' => $entity->bundle(),
          'id' => $entity->id(),
        ];
      }
    }
    return $validations;
  }

  /**
   * {@inheritdoc}
   */
  public function validateTextField(string $field_name, EntityInterface $entity): array {
    $validations = [];
    if (AtvCommonTools::isTextImageField($field_name, $entity)) {
      $field_values = $entity->get($field_name)->getValue();
      foreach ($field_values as $key => $field_value) {
        $html_string = $field_value['value'] ?? '';
        $images = $this->extractImageTags($html_string);
        foreach ($images as $image) {
          $file_source = $image['src'];
          $file_name = pathinfo($file_source, PATHINFO_BASENAME);
          $apex_entity = $this->getApexEntity($entity);
          $validations[] = [
            'apex_entity_title' => $apex_entity->label(),
            'apex_entity_url' => ($apex_entity->isNew()) ? '' : $apex_entity->toUrl('canonical', ['absolute' => TRUE])->toString(),
            'field_name' => $field_name,
            'file_name' => $file_name,
            'file_source' => $file_source,
            'title' => $image['title'],
            'alt' => $image['alt'],
            'violations' => $this->getViolations($file_name, $image['alt'], $image['title']),
            'warnings' => $this->getWarnings($file_name, $image['alt'], $image['title']),
            'entity_type' => $entity->getEntityTypeId(),
            'bundle' => $entity->bundle(),
            'id' => $entity->id(),
          ];
        }
      }
    }

    return $validations;
  }

  /**
   * Extract images from a string of HTML.
   *
   * @param string $html_string
   *   The sting to extract images from.
   *
   * @return array
   *   Array of arrays each containing src, alt, and title elements.
   */
  public static function extractImageTags(string $html_string):array {
    $images = [];
    if (empty($html_string)) {
      return $images;
    }
    // We are only loading a snippet of html so we need to wrap it to make it
    // look like a complete page.
    $html_string = "<!DOCTYPE html><html><body>$html_string</body></html>";
    $dom = new \DOMDocument();
    // The contents of the text field may trigger some DomDocument warnings
    // that we don't need to see. So we silence them temporarily.
    libxml_use_internal_errors(TRUE);
    $dom->loadHtml($html_string);
    libxml_use_internal_errors(FALSE);
    foreach ($dom->getElementsByTagName('img') as $element) {
      $images[] = [
        'src' => $element->getAttribute('src'),
        'alt' => $element->getAttribute('alt'),
        'title' => $element->getAttribute('title'),
      ];
    }
    return $images;
  }

  /**
   * Get all the enabled rules.
   *
   * @return \Drupal\alt_text_validation\Entity\AltTextRuleInterface[]
   *   An array of rule entities.
   */
  public function getRules(): array {
    if (empty($this->rules)) {
      $ruleStorage = $this->entityTypeManager->getStorage('alt_text_rule');
      // We need to load all the rules that are not off.
      $ids = $ruleStorage->getQuery()
        ->condition('violation_action', 'off', '<>')
        // We need the rules regardless of who can see them.
        ->accessCheck(FALSE)
        ->execute();
      $this->rules = $ruleStorage->loadMultiple($ids);
    }
    return $this->rules;
  }

  /**
   * Get all the enabled prevention rules (strict).
   *
   * @return \Drupal\alt_text_validation\Entity\AltTextRuleInterface[]
   *   An array of rule entities that prevent save.
   */
  public function getPreventionRules(): array {
    if (is_null($this->preventionRules)) {
      $prevent_rules = [];
      foreach ($this->getRules() as $rule) {
        if ($rule->getViolationAction() === 'prevent') {
          $prevent_rules[] = $rule;
        }
      }
      $this->preventionRules = $prevent_rules;
    }

    return $this->preventionRules;
  }

  /**
   * Get all the enabled warning rules.
   *
   * @return \Drupal\alt_text_validation\Entity\AltTextRuleInterface[]
   *   An array of rule entities that are warning rules.
   */
  public function getWarningRules(): array {
    if (is_null($this->warningRules)) {
      $warning_rules = [];
      foreach ($this->getRules() as $rule) {
        if ($rule->getViolationAction() === 'warn') {
          $warning_rules[] = $rule;
        }
      }
      $this->warningRules = $warning_rules;
    }

    return $this->warningRules;
  }

  /**
   * Recursively get the apex entity that hosts this entity.
   *
   * If this is a paragraph, it will get the parent entity of
   * the paragraph. If that parent is also a paragraph, it will continue
   * until it finds an entity that is not a paragraph (or other nested
   * entity) and return that.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to start from.
   *
   * @return \Drupal\Core\Entity\EntityInterface
   *   The apex entity that hosts this entity.
   */
  protected function getApexEntity(EntityInterface $entity): EntityInterface {
    // Check if entity has a way to get parents, like Paragraphs do.
    if (method_exists($entity, 'getParentEntity')) {
      $parent = $entity->getParentEntity();
      if ($parent) {
        // There is a parent, so see if there is a grandparent.
        return $this->getApexEntity($parent);
      }
    }
    // Check if entity has a way to get Entity, like file entities do.
    if (method_exists($entity, 'getEntity')) {
      $parent = $entity->getEntity();
      if ($parent) {
        // There is a parent, so see if there is a grandparent.
        return $this->getApexEntity($parent);
      }
    }
    // Otherwise, return the entity itself.
    return $entity;
  }

}
