<?php

namespace Drupal\alt_text_validation\Entity;

use Drupal\Core\Config\Entity\ConfigEntityInterface;

/**
 * Interface for Text Rule Configuration entities.
 *
 * Provides an interface defining a text rule entity type.
 */
interface AltTextRuleInterface extends ConfigEntityInterface {

  /**
   * The available rule type options with human-readable labels.
   */
  const RULE_TYPE_OPTIONS = [
    'alt_is_filename' => 'Cannot match image filename',
    'alt_is_title' => 'Cannot match image title',
    'not_empty' => 'Cannot be empty unless decorative',
    'length_limit' => 'Cannot exceed length',
    'not_begin_with' => 'Cannot begin with',
    'not_equal' => 'Cannot equal',
    'not_contain' => 'Cannot contain',
    'not_end_with' => 'Cannot end with',
    'regex_match' => 'Cannot match regex',
  ];

  /**
   * Sets the Text Rule type.
   *
   * @param string $rule_type
   *   The Text Rule type.
   *
   * @return $this
   *   The called Text Rule entity.
   */
  public function setRuleType(string $rule_type): AltTextRuleInterface;

  /**
   * Gets the Text Rule type.
   *
   * @return ?string
   *   The Text Rule type or NULL.
   */
  public function getRuleType(): ?string;

  /**
   * Gets the human-readable label for the rule type.
   *
   * @return string
   *   The translated human-readable label for the rule type.
   */
  public function getRuleTypeLabel();

  /**
   * Gets the rule type options.
   *
   * @return array
   *   The array of rule type options.
   */
  public function getRuleTypeOptions();

  /**
   * Sets the content character limit.
   *
   * @param int $content_char_limit
   *   The content character limit.
   *
   * @return $this
   *   The called Text Rule entity.
   */
  public function setContentCharLimit(int $content_char_limit): AltTextRuleInterface;

  /**
   * Gets the content character limit, or NULL if uninitialized.
   *
   * @return ?int
   *   The content character limit or NULL.
   */
  public function getContentCharLimit(): ?int;

  /**
   * Sets the text to compare.
   *
   * @param string $text_to_compare
   *   The text to compare.
   *
   * @return $this
   *   The called Text Rule entity.
   */
  public function setTextToCompare(string $text_to_compare): AltTextRuleInterface;

  /**
   * Gets the text to compare, or NULL if uninitialized.
   *
   * @return ?string
   *   The text to compare or NULL.
   */
  public function getTextToCompare(): ?string;

  /**
   * Sets the violation action.
   *
   * @param string $violation_action
   *   The violation action.
   *
   * @return $this
   *   The called Text Rule entity.
   */
  public function setViolationAction(string $violation_action): AltTextRuleInterface;

  /**
   * Gets the violation action, or NULL if uninitialized.
   *
   * @return ?string
   *   The violation action.
   */
  public function getViolationAction(): ?string;

  /**
   * Sets the violation message.
   *
   * @param string $violation_message
   *   The violation message.
   *
   * @return $this
   *   The called Text Rule entity.
   */
  public function setViolationMessage(string $violation_message): AltTextRuleInterface;

  /**
   * Gets the violation message.
   *
   * @return string
   *   The violation message.
   */
  public function getViolationMessage(): ?string;

  /**
   * Applies the rule to the given alt text and title.
   *
   * @param string $filename
   *   The file name of the image file.
   * @param string $alt
   *   The alt text of the image.
   * @param string $title
   *   The title text of the image.
   *
   * @return bool
   *   TRUE if the rule is violated, FALSE otherwise.
   */
  public function isViolation(string $filename, string $alt, string $title): bool;

}
