<?php

namespace Drupal\alt_text_validation\Plugin\Validation\Constraint;

use Drupal\alt_text_validation\AtvCommonTools;
use Drupal\alt_text_validation\Service\ValidationToolsInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Render\Renderer;
use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\ConstraintValidator;

/**
 * Validates the UniqueInteger constraint.
 */
class AltTextRulesConstraintValidator extends ConstraintValidator implements ContainerInjectionInterface {

  /**
   * The renderer.
   *
   * @var \Drupal\Core\Render\Renderer
   */
  protected $renderer;


  /**
   * The validation tools service.
   *
   * @var \Drupal\alt_text_validation\Service\ValidationToolsInterface
   */
  protected $validationTools;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The configuration factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a AltTextRulesConstraintValidator object.
   *
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\alt_text_validation\Service\ValidationToolsInterface $validation_tools
   *   The validation tools service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory service.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer.
   */
  public function __construct(MessengerInterface $messenger, ValidationToolsInterface $validation_tools, ConfigFactoryInterface $config_factory, Renderer $renderer) {
    $this->messenger = $messenger;
    $this->validationTools = $validation_tools;
    $this->configFactory = $config_factory;
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create($container) {
    return new static(
      $container->get('messenger'),
      $container->get('alt_text_validation.validationtools'),
      $container->get('config.factory'),
      $container->get('renderer')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function validate($value, Constraint $constraint) {
    $config = $this->configFactory->get('alt_text_validation.settings');
    $altTextValidationEnabled = $config->get('alt_text_validation_enabled') ?? 0;
    // Skip validation if alt_text_validation_enabled is off (0).
    if (($altTextValidationEnabled == 0) || $this->isAjaxRequest()) {
      return;
    }

    $image_validations = $this->getValidations($value);
    if (!empty($image_validations)) {
      $errors = '';
      $warning_count = '0';
      foreach ($image_validations as $image_validation) {
        if (!empty($image_validation['violations'])) {
          // Address the violations for this image.
          $list = $this->buildMessageList($image_validation['violations']);
          // A heading is only needed if it is the first error for the field.
          $heading = (empty($errors)) ? $config->get('heading_prevent') ?? '' : '';
          $render_array = $this->buildRenderArray(
            'prevent',
            $heading,
            $image_validation['file_name'],
            $image_validation['alt'],
            $value->getFieldDefinition()->getLabel(),
            $list
          );
          $errors .= $this->renderer->renderInIsolation($render_array);
        }

        if (!empty($image_validation['warnings'])) {
          // Address the warnings.
          $warning_list = $this->buildMessageList($image_validation['warnings']);
          // A heading is only needed if it is the first error for the field.
          (!empty($warning_list)) ? $warning_count++ : '';
          $warn_heading = ($warning_count == 1) ? $config->get('heading_warn') ?? '' : '';
          $warning_render_arrays = $this->buildRenderArray(
            'warn',
            $warn_heading,
            $image_validation['file_name'],
            $image_validation['alt'],
            $value->getFieldDefinition()->getLabel(),
            $warning_list
          );

          // If you came directly from a View using an "edit" operation, then
          // when the save completes, you will likely get redirected back to
          // that View. Unfortunately the user may lose the context of what
          // generated the warning. A link to the edit page or node may help.
          // @todo Add a link to the page generating the warning
          // $entity = $value->getParent()->getEntity();
          // $absolute_url_string = $entity->toUrl('canonical',
          // ['absolute' => TRUE])->toString();
          if (!empty($warning_render_arrays)) {
            $this->messenger->addWarning($warning_render_arrays);
          }

        }
      }
      $this->throwViolation($errors);
    }

  }

  /**
   * Build a render array for the alt text validation components.
   *
   * @param string $type
   *   The type of message (warn/prevent).
   * @param string $heading
   *   The heading to use.
   * @param string $filename
   *   The image filename.
   * @param string $alt
   *   The alt text.
   * @param string $label
   *   The field label.
   * @param array $messagelist
   *   A renderable array of messages.
   *
   * @return array
   *   A renderable array for the components
   */
  protected function buildRenderArray(string $type, string $heading, string $filename, string $alt, string $label, array $messagelist): array {
    $render_array = [
      '#type' => 'component',
      '#component' => ($type === 'warn') ? 'alt_text_validation:atv-warning' : 'alt_text_validation:atv-prevent',
      '#slots' => [
        'heading' => $heading,
        'filename' => $filename,
        'alt_text' => $alt,
        'field_label' => $label,
        'messages' => $messagelist,
      ],
    ];
    return $render_array;
  }

  /**
   * Get the validations for the given field value.
   *
   * @param \Drupal\Core\Field\FieldItemListInterface $value
   *   The field value.
   *
   * @return array
   *   An array of validations for a single field.
   */
  protected function getValidations(FieldItemListInterface $value): array {
    $is_image = AtvCommonTools::isImageField($value->getName(), $value->getEntity());
    if ($is_image) {
      return $this->validationTools->validateImageField($value->getName(), $value->getEntity());
    }
    else {
      return $this->validationTools->validateTextField($value->getName(), $value->getEntity());
    }
  }

  /**
   * Convert array messages into renderable list of #markup.
   *
   * @param array $messages
   *   The messages to convert (warnings or errors).
   *
   * @return array
   *   A renderable array of messages.
   */
  protected function buildMessageList(array $messages): array {
    $list = [];
    foreach ($messages as $message) {
      $list[]['#markup'] = $message;
    }
    return $list;
  }

  /**
   * Throw a violation with the given message.
   *
   * @param string $message_html
   *   The message HTML to use.
   */
  protected function throwViolation(string $message_html): void {
    // @todo Fix the jump-link to the erroring field not showing up.
    if (!empty($message_html)) {
      $this->context->buildViolation($message_html)
        ->addViolation();
    }
  }

  /**
   * Determine if the current request is an AJAX request.
   *
   * @return bool
   *   TRUE if it is an AJAX request, FALSE otherwise.
   */
  protected function isAjaxRequest(): bool {
    if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest') {
      return TRUE;
    }
    return FALSE;
  }

}
