<?php

namespace Drupal\alt_text_validation\Plugin\views\filter;

use Drupal\alt_text_validation\Service\ValidationToolsInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\views\Attribute\ViewsFilter;
use Drupal\views\Plugin\views\display\DisplayPluginBase;
use Drupal\views\Plugin\views\filter\InOperator;
use Drupal\views\ViewExecutable;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Filter handler for my custom field.
 *
 * @ingroup views_filter_handlers
 */
#[ViewsFilter("atv_rule_filter")]
class AtvRuleFilter extends InOperator implements ContainerFactoryPluginInterface {

  /**
   * The SQL query interface.
   *
   * @var \Drupal\views\Plugin\views\query\Sql
   */
  public $query;

  /**
   * The collection of Rule entities.
   *
   * @var array
   */
  protected $rules;

  /**
   * The validation tools service.
   *
   * @var \Drupal\alt_text_validation\Service\ValidationToolsInterface
   */
  protected $validationTools;

  /**
   * Constructs a new YourCustomFilter object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\alt_text_validation\Service\ValidationToolsInterface $validation_tools
   *   The validation tools service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, ValidationToolsInterface $validation_tools) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->validationTools = $validation_tools;

  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('alt_text_validation.validationtools')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function init(ViewExecutable $view, DisplayPluginBase $display, ?array &$options = NULL) {
    parent::init($view, $display, $options);
    $this->no_operator = TRUE;
    $this->valueFormType = 'select';
    // Set the default value to 'All' to show all rules.
    $this->value = ['All'];
  }

  /**
   * {@inheritdoc}
   */
  public function getValueOptions() {
    // Define your dropdown options here.
    if (isset($this->valueOptions)) {
      return $this->valueOptions;
    }
    $this->valueOptions = [];
    $rules = $this->getRuleValues();
    // Provide a fallback in case no rules are found.
    $this->valueOptions = (!empty($rules) ? $rules : []);

    return $this->valueOptions;
  }

  /**
   * Gets all the active Rules and returns them as an associative array.
   *
   * @return array
   *   An associative array of distinct values.
   */
  protected function getRuleValues(): array {
    $rules = $this->getRules();
    $values = [];
    foreach ($rules as $rule) {
      $values[$rule->id] = $rule->label();
    }

    asort($values);
    return $values;
  }

  /**
   * Gets the label for a specific rule ID.
   *
   * @param string $rule_id
   *   The rule ID to get the label for.
   *
   * @return string
   *   The label of the rule.
   */
  protected function getRuleLabel(string $rule_id): string {
    $rules_values = $this->getRuleValues();
    $rule_label = $rules_values[$rule_id] ?? '';
    return $rule_label;
  }

  /**
   * Filters by a simple operator because there are no operators.
   */
  protected function opSimple(): void {
    if (empty($this->value) || ($this->value[0] === 'All')) {
      // If no value is set or 'All' is selected, do not filter.
      return;
    }
    $this->ensureMyTable();
    $rule_id = $this->value[0];
    $rule_label = $this->getRuleLabel($rule_id);
    // Search for the html markup to make sure one rule label is not a short
    // version of another one. That would lead to a false positive.
    $search_value = "<li>{$rule_label}</li>";

    // Check if 'warn' or 'prevent' rule to figure out the appropriate column.
    $column = ($this->isPreventRule($rule_id)) ? 'violation_list' : 'warning_list';
    $this->query->addWhere($this->options['group'], $column, "%{$search_value}%", 'LIKE');
  }

  /**
   * Checks if the rule is a 'prevent' rule.
   *
   * @param string $rule_id
   *   The rule ID to check.
   *
   * @return bool
   *   TRUE if the rule is a 'prevent' rule, FALSE otherwise.
   */
  protected function isPreventRule(string $rule_id): bool {
    $rules = $this->getRules();
    foreach ($rules as $rule) {
      if (($rule->id === $rule_id) && ($rule->getViolationAction() === 'prevent')) {
        return TRUE;
      }
    }
    return FALSE;
  }

  /**
   * Gets all the enabled rules.
   *
   * @return \Drupal\alt_text_validation\Entity\AltTextRuleInterface[]
   *   An array of rule entities.
   */
  public function getRules(): array {
    if (empty($this->rules)) {
      $this->rules = $this->validationTools->getRules();
    }
    return $this->rules;
  }

}
