<?php

namespace Drupal\alt_text_validation\Service;

use Drupal\alt_text_validation\AtvCommonTools;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\State\State;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class for Auditor to process and build the audit.
 */
class Auditor implements AuditorInterface, ContainerInjectionInterface {

  use DependencySerializationTrait;
  use StringTranslationTrait;

  /**
   * The configuration for alt_text_validation.
   *
   * @var \Drupal\Core\Config\Config
   */
  protected $atvConfig;

  /**
   * The Alt Text Validation audit storage service.
   *
   * @var \Drupal\alt_text_validation\Service\AuditStorageInterface
   */
  public $auditStorage;

  /**
   * The Database connector service.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $databaseConnection;

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The alt_text_validation logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  public $logger;

  /**
   * The queue service.
   *
   * @var \Drupal\Core\Queue\QueueFactory
   */
  protected $queue;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\State
   */
  public $state;

  /**
   * Constructs the AuditStorage service.
   *
   * @param Drupal\alt_text_validation\Service\AuditStorageInterface $audit_storage
   *   The audit storage service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param Drupal\Core\Database\Connection $database_connection
   *   The database connection service.
   * @param Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager service.
   * @param Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   * @param Psr\Log\LoggerInterface $logger
   *   The alt_text_validation logger.
   * @param Drupal\Core\Queue\QueueFactory $queue
   *   The queue service.
   * @param Drupal\Core\State\State $state
   *   The state service.
   */
  final public function __construct(
    AuditStorageInterface $audit_storage,
    ConfigFactoryInterface $config_factory,
    Connection $database_connection,
    EntityFieldManagerInterface $entity_field_manager,
    EntityTypeManagerInterface $entity_type_manager,
    LoggerInterface $logger,
    QueueFactory $queue,
    State $state,
  ) {
    $this->auditStorage = $audit_storage;
    $this->atvConfig = $config_factory->getEditable('alt_text_validation.settings');
    $this->databaseConnection = $database_connection;
    $this->entityFieldManager = $entity_field_manager;
    $this->entityTypeManager = $entity_type_manager;
    $this->logger = $logger;
    $this->queue = $queue;
    $this->state = $state;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('alt_text_validation.audit_storage'),
      $container->get('config.factory'),
      $container->get('database'),
      $container->get('entity_field.manager'),
      $container->get('entity_type.manager'),
      $container->get('logger.channel.alt_text_validation'),
      $container->get('queue'),
      $container->get('state')
    );
  }

  /**
   * Collects all field instances that might contain alt text.
   *
   * @return array
   *   An array of fields that might contain alt text.
   */
  protected function collectImageContainingFields(): array {
    $image_instances = [];
    $entity_types = $this->entityFieldManager->getFieldMap();
    // Loop through retrieved field instances and access relevant information.
    foreach ($entity_types as $entity_type => $entity_fields) {
      foreach ($entity_fields as $entity_field => $field_info) {
        if (AtvCommonTools::isAltContainingTypeField($field_info['type'])) {
          foreach ($field_info['bundles'] as $bundle) {
            $image_instances[$entity_type][$bundle][] = $entity_field;
          }
        }
      }
    }
    return $image_instances;
  }

  /**
   * {@inheritdoc}
   */
  public function queueAllImages(): void {
    // We want to reset the audit table and any items still in queues.
    $this->auditStorage->truncateTable();
    $this->truncateQueues();
    $states = [
      AtvCommonTools::AUDIT_STATUS_KEY => $this->t('Audit in progress.'),
      AtvCommonTools::AUDIT_START_TIME_KEY => time(),
    ];
    $this->state->setMultiple($states);
    $image_containing_entities = $this->collectImageContainingFields();
    $atvqueue = $this->queue->get(AtvCommonTools::ENTITY_TYPE_QUEUE_NAME);
    foreach ($image_containing_entities as $entity_type => $bundles) {
      foreach ($bundles as $bundle => $fields) {
        $item = new \stdClass();
        $item->entity_type = $entity_type;
        $item->bundle = $bundle;
        $item->fields = $fields;
        $atvqueue->createItem($item);
        $vars = [
          '@entity_type' => $entity_type,
          '@bundle' => $bundle,
        ];
        $this->logger->info('Queued entity type: @entity_type bundle: @bundle', $vars);
      }
    }
    $item_count = $atvqueue->numberOfItems();
    $vars = [
      '@queue_name' => AtvCommonTools::ENTITY_TYPE_QUEUE_NAME,
      '@count' => $item_count,
    ];
    $this->logger->info('Queued @count entity:bundles that have fields that could support images into @queue_name.', $vars);
  }

  /**
   * Truncates all alt_text_validation queues.
   */
  protected function truncateQueues(): void {
    $this->queue->get(AtvCommonTools::ENTITY_TYPE_QUEUE_NAME)->deleteQueue();
    $this->queue->get(AtvCommonTools::ENTITY_INSTANCE_QUEUE_NAME)->deleteQueue();
  }

  /**
   * {@inheritdoc}
   */
  public function tryCron(): void {
    // Check settings to see if cron processing is enabled.
    if (!empty($this->atvConfig->get('cron_enabled'))) {
      // Check the last completed time.
      $times = $this->state->getMultiple([
        AtvCommonTools::AUDIT_START_TIME_KEY,
        AtvCommonTools::AUDIT_END_TIME_KEY,
      ]);
      $running = !empty($times[AtvCommonTools::AUDIT_START_TIME_KEY]) && empty($times[AtvCommonTools::AUDIT_END_TIME_KEY]);
      $delay_is_right = $this->checkDelay($times[AtvCommonTools::AUDIT_END_TIME_KEY]);
      if ((!$running && $delay_is_right) || $this->checkTooLong($times[AtvCommonTools::AUDIT_START_TIME_KEY])) {
        $this->queueAllImages();
      }
    }
  }

  /**
   * Checks if the delay between audits is sufficient.
   *
   * @param int|null $last_end_time
   *   The timestamp of the last completed audit.
   *
   * @return bool
   *   TRUE if the delay was sufficient, FALSE otherwise.
   */
  protected function checkDelay(int|null $last_end_time): bool {
    if (empty($last_end_time)) {
      // It was not last run, so it is ready to run.
      return TRUE;
    }
    $delay = $this->atvConfig->get('audit_delay') * 24 * 60 * 60;
    return (time() - $last_end_time) >= $delay;
  }

  /**
   * Checks if it has been too long since run started, or never run.
   *
   * @param int|null $last_start_time
   *   The timestamp of the last completed audit.
   *
   * @return bool
   *   TRUE if the delay was sufficient, FALSE otherwise.
   */
  protected function checkTooLong(int|null $last_start_time): bool {
    if (empty($last_start_time)) {
      // It has never run, so it is ready to run.
      return TRUE;
    }
    // We will consider an additional 2 days beyond should mean something is
    // stuck so run this again.
    $delay = ($this->atvConfig->get('audit_delay') + 2) * 24 * 60 * 60;
    return (time() - ($last_start_time)) >= $delay;
  }

}
