<?php

namespace Drupal\alt_text_validation\Service;

use Drupal\alt_text_validation\AtvCommonTools;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * The batchprocessor for rebuilding the audit report contents.
 */
class BatchProcessor {

  use DependencySerializationTrait;
  use StringTranslationTrait;

  /**
   * Use this constant to set # of queued items to process in one batch cycle.
   */
  const AUDIT_BATCH_SIZE = 1;

  /**
   * The auditor service.
   *
   * @var \Drupal\alt_text_validation\Service\Auditor
   */
  protected $auditor;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The queue service.
   *
   * @var \Drupal\Core\Queue\QueueFactory
   */
  protected $queue;

  /**
   * The plugin manager for queue workers.
   *
   * @var \Drupal\Core\Plugin\DefaultPluginManager
   */
  protected $queueWorkerManager;

  /**
   * Constructs the BatchProcessor service.
   *
   * @param \Drupal\alt_text_validation\Service\Auditor $auditor
   *   The auditor service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\Queue\QueueFactory $queue
   *   The queue service.
   * @param \Drupal\Core\Plugin\DefaultPluginManager $queueWorkerManager
   *   The queue worker manager service.
   */
  public function __construct(Auditor $auditor, MessengerInterface $messenger, QueueFactory $queue, DefaultPluginManager $queueWorkerManager) {
    $this->auditor = $auditor;
    $this->messenger = $messenger;
    $this->queue = $queue;
    $this->queueWorkerManager = $queueWorkerManager;
  }

  /**
   * Build the queues and trigger processing.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse|null
   *   Redirect response object.
   */
  public function batchProcess(): RedirectResponse {
    // Create the queue.
    $this->auditor->queueAllImages();

    $batch = [
      'title' => $this->t('Alt Text audit queues with batch'),
      'operations' => [],
      'finished' => [$this, 'batchFinished'],
    ];
    $atvqueue = $this->queue->get(AtvCommonTools::ENTITY_TYPE_QUEUE_NAME);

    // Count items in this queue, and create enough batch operations.
    for ($i = 0; $i < ceil($atvqueue->numberOfItems() / self::AUDIT_BATCH_SIZE); $i++) {
      // Create batch operations.
      $batch['operations'][] = [[$this, 'processQueue'], []];
    }
    // Adds the batch sets.
    batch_set($batch);
    // Process the batch and after redirect to the report page.
    return batch_process('/admin/reports/alt-text-report');
  }

  /**
   * Processes the two queues in sequence.
   *
   * @param mixed $context
   *   The context for the operation.
   */
  public function processQueue(&$context): void {
    // Get the queue implementation for atv_entity_types queue.
    $entity_type_queue = $this->queue->get(AtvCommonTools::ENTITY_TYPE_QUEUE_NAME);
    // Get the queue worker.
    $queue_worker = $this->queueWorkerManager->createInstance(AtvCommonTools::ENTITY_TYPE_QUEUE_NAME);

    // Get the number of items.
    $number_of_queue = ($entity_type_queue->numberOfItems() < self::AUDIT_BATCH_SIZE) ? $entity_type_queue->numberOfItems() : self::AUDIT_BATCH_SIZE;
    for ($i = 0; $i < $number_of_queue; $i++) {

      // Get a queued item.
      if ($item = $entity_type_queue->claimItem()) {

        // Process it.
        $queue_worker->processItem($item->data);

        // If everything was correct, delete the processed item from the queue.
        $entity_type_queue->deleteItem($item);

        // Get the queue implementation for atv_entity_instances queue.
        $atv_entity_instances_queue = $this->queue->get(AtvCommonTools::ENTITY_INSTANCE_QUEUE_NAME);
        // Get the queue worker.
        $atv_entity_instances_queue_worker = $this->queueWorkerManager->createInstance(AtvCommonTools::ENTITY_INSTANCE_QUEUE_NAME);
        for ($j = 0; $j < $atv_entity_instances_queue->numberOfItems(); $j++) {
          if ($atv_entity_instances_queue_item = $atv_entity_instances_queue->claimItem()) {
            $atv_entity_instances_queue_worker->processItem($atv_entity_instances_queue_item->data);
            $atv_entity_instances_queue->deleteItem($atv_entity_instances_queue_item);
          }
        }
      }
    }
  }

  /**
   * Batch finished callback.
   */
  public function batchFinished($success, $results, $operations): void {
    if ($success) {
      $this->messenger->addMessage($this->t('Alt Text audit completed successfully.'));
    }
    else {
      $this->messenger->addMessage($this->t('Batch process encountered an error.'), 'error');
    }
  }

}
