<?php

namespace Drupal\altcolor\Hook;

use Drupal\altcolor\Plugin\AltColorPluginManagerInterface;
use Drupal\altcolor\Form\ColorForm;
use Drupal\Core\DependencyInjection\ClassResolverInterface;
use Drupal\Core\Form\FormState;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;

/**
 * Form implementations for the Alternative color module.
 */
class AltColorFormHooks {

  /**
   * Constructor for AltColorFormHooks.
   *
   * @param \Drupal\altcolor\Plugin\AltColorPluginManagerInterface $alternativeColorManager
   *   The alternative color manager.
   * @param \Drupal\Core\DependencyInjection\ClassResolverInterface $classResolver
   *   The class resolver.
   */
  public function __construct(
    protected AltColorPluginManagerInterface $alternativeColorManager,
    protected ClassResolverInterface $classResolver,
  ) {
  }

  /**
   * Implements hook_form_FORM_ID_alter() for the system_theme_settings form.
   *
   * This hook is used to embed the altcolor form into the theme settings form.
   */
  #[Hook('form_system_theme_settings_alter')]
  public function formSystemThemeSettingsAlter(array &$form, FormStateInterface $form_state, string $form_id) {
    $build_info = $form_state->getBuildInfo();
    // Check that we are looking at a theme config page and not global config.
    if (isset($build_info['args'][0]) && ($theme_name = $build_info['args'][0])) {
      $theme_colors_definition = $this->alternativeColorManager->getColorDefinitionsByTheme($theme_name);
      // Only do something if the theme has defined colors in THEME.colors.yml.
      if (!empty($theme_colors_definition) && $theme_colors_definition->getColors()) {
        /** @var \Drupal\altcolor\Form\ColorForm $color_form */
        $color_form = $this->classResolver->getInstanceFromDefinition(ColorForm::class);
        $new_form_state = (new FormState());
        $form += $color_form->buildForm([], $new_form_state, $theme_name);
        // Set this module's submit handler to run first, so that the values can
        // be moved to the third party settings before the config is saved.
        array_unshift($form['#submit'], [$this, 'formSystemThemeSettingsSubmit']);
      }
    }
  }

  /**
   * Additional form submit handler for the system_theme_settings form.
   *
   * The original form submit handler for the system_theme_settings form already
   * saves all form values. By reordering the form values, we can move our
   * values to the third party settings prior to the theme config getting saved
   * to the database.
   *
   * @see \Drupal\system\Form\ThemeSettingsForm::submitForm()
   * @see theme_settings_convert_to_config()
   */
  public function formSystemThemeSettingsSubmit(array $form, FormStateInterface $form_state) {
    // @todo Decide if we should empty the array when the form is submitted with
    //   default values. In that case no values will be injected into the page
    //   and the default CSS will be used instead.
    // @todo 'third_party_settings in THEME.settings.yml cannot be dependency
    //   managed': https://www.drupal.org/project/drupal/issues/3087036
    //   When/if themes support third party settings, revise this.
    //   $config = \Drupal::configFactory()->getEditable($theme . '.config');
    //   $config->setThirdPartySetting('altcolor', 'colors', $form_state
    //   ->getValue('altcolor'));
    $form_values = $form_state->getValues();
    // Move the altcolor values to the third party settings.
    $form_values['third_party_settings']['altcolor']['colors'] = $form_values['altcolor']['colors'];
    $form_state->setValues($form_values);

    // Unset the altcolor key to prevent it from being saved in the theme
    // config.
    $form_state->unsetValue('altcolor');
  }

}
