/**
 * @file
 * Handles the generation of alt text for images.
 */

(function ($, Drupal, once, drupalSettings) {
  'use strict';

  // Configuration
  const CONFIG = {
    pollInterval: 3000, // 3 seconds
    maxPollAttempts: 20,
    endpoints: drupalSettings.alttexting.endpoints,
    messages: {
      generating: Drupal.t('Generating...'),
      success: Drupal.t('Alt text generated successfully!'),
      error: Drupal.t('Failed to generate alt text. Please check your configuration and try again.'),
      timeout: Drupal.t('Request timed out. Please try again.')
    }
  };

  /**
   * Handles the alt text generation process.
   * @type {{attach: Drupal.behaviors.altTextGenerator.attach}}
   */
  Drupal.behaviors.altTextGenerator = {
    attach: function (context, settings) {
      once('alt-text-generator', '.generate-alt-button', context).forEach(button => {
        const $button = $(button);
        $button.on('click', this.handleButtonClick.bind(this, $button));
      });
    },

    /**
     * Handles the generate button click event.
     * @param {jQuery} $button - The clicked button element.
     * @param {Event} event - The click event.
     */
    handleButtonClick: function ($button, event) {
      event.preventDefault();

      const fid = $button.data('image-fid');
      const langcode = $button.data('image-langcode');

      if (!fid) {
        this.showError($button, Drupal.t('Missing image ID'));
        return;
      }

      const $altField = $button.closest('.js-form-item').find('input[name*="[alt]"]');
      this.setButtonState($button, true);

      this.startGeneration(fid, langcode, $button)
        .then(response => this.pollForResult(response.id, $button, $altField))
        .catch(error => {
          this.showError($button, error.message || CONFIG.messages.error);
        });
    },

    /**
     * Initiates the alt text generation process.
     * @param {string} fid - The file ID of the image.
     * @returns {Promise<Object>} Promise that resolves with the generation response.
     */
    startGeneration: function (fid, langcode, $button) {
      return new Promise((resolve, reject) => {
        $.ajax({
          url: CONFIG.endpoints.generate,
          method: 'POST',
          data: {fid: fid, langcode: langcode},
          timeout: 10000 // 10 second timeout
        })
          .done(response => resolve(response))
          .fail((jqXHR, textStatus) => {
            this.showError($button, textStatus === 'timeout' ? CONFIG.messages.timeout : CONFIG.messages.error);
            reject(new Error(textStatus === 'timeout' ? CONFIG.messages.timeout : textStatus));
          });
      });
    },

    /**
     * Polls for the generation result.
     * @param {string} id - The generation ID.
     * @param {jQuery} $button - The button element.
     * @param {jQuery} $altField - The alt text input field.
     * @param {number} [attempt=1] - Current attempt number.
     */
    pollForResult: function (id, $button, $altField, attempt = 1) {
      if (attempt > CONFIG.maxPollAttempts) {
        throw new Error(Drupal.t('Maximum polling attempts reached'));
      }

      return new Promise((resolve, reject) => {
        const poll = () => {
          $.ajax({
            url: CONFIG.endpoints.checkResult,
            data: {id},
            timeout: 10000
          })
            .done(response => {
              if (response.hasAnswer && response.altText) {
                this.handleSuccess($button, $altField, response.altText);
                resolve();
              } else if (response.error) {
                this.showError($button, CONFIG.messages.error);
                console.log(response.error);
                reject(new Error(response.error));
              } else if (attempt < CONFIG.maxPollAttempts) {
                setTimeout(() => {
                  this.pollForResult(id, $button, $altField, attempt + 1);
                }, CONFIG.pollInterval);
              } else {
                this.showError($button, CONFIG.messages.timeout);
                reject(new Error(Drupal.t('Generation timeout')));
              }
            })
            .fail(() => {
              if (attempt < CONFIG.maxPollAttempts) {
                setTimeout(poll, CONFIG.pollInterval * Math.pow(1.5, attempt));
              } else {
                reject(new Error(Drupal.t('Failed to check generation status')));
              }
            });
        };

        poll();
      });
    },

    /**
     * Handles successful alt text generation.
     * @param {jQuery} $button - The button element.
     * @param {jQuery} $altField - The alt text input field.
     * @param {string} altText - The generated alt text.
     */
    handleSuccess: function ($button, $altField, altText) {
      $altField.val(altText).trigger('change');
      this.setButtonState($button, false);

      // Optional: Show a success message that fades out
      const $success = $('<div class="messages messages--status">' + CONFIG.messages.success + '</div>');
      $button.after($success);
      setTimeout(() => $success.fadeOut(1000, () => $success.remove()), 3000);
    },

    /**
     * Shows an error message.
     * @param {jQuery} $button - The button element.
     * @param {string} message - The error message.
     */
    showError: function ($button, message) {
      this.setButtonState($button, false, message);

      // Optional: Show error message
      const $error = $('<div class="messages messages--error">' + message + '</div>');
      $button.after($error);
      setTimeout(() => $error.fadeOut(1000, () => $error.remove()), 5000);
    },

    /**
     * Updates the button state.
     * @param {jQuery} $button - The button element.
     * @param {boolean} isLoading - Whether the button is in loading state.
     */
    setButtonState: function ($button, isLoading) {
      const $buttonText = $button.find('.sparkle-btn_text');
      const originalText = $button.data('original-text') || $buttonText.text();

      if (isLoading) {
        $button.data('original-text', originalText);
        $button.prop('disabled', true).addClass('sparkling');
        $buttonText.text(CONFIG.messages.generating);
      } else {
        $button.prop('disabled', false).removeClass('sparkling');
        $buttonText.text($button.data('original-text') || originalText);
      }
    }
  };

})(jQuery, Drupal, once, drupalSettings);
