<?php

namespace Drupal\alttexting\Controller;

use Drupal\alttexting\Service\AltTextGeneratorService;
use Drupal\Core\Controller\ControllerBase;
use Exception;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Controller for handling webhook requests for alternative text generation.
 */
class WebhookController extends ControllerBase {

  /**
   * The alt text generator service.
   *
   * @var \Drupal\alttexting\Service\AltTextGeneratorService
   */
  protected AltTextGeneratorService $altTextGenerator;

  /**
   * The logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * Constructs a new WebhookController object.
   *
   * @param \Drupal\alttexting\Service\AltTextGeneratorService $alt_text_generator
   *   The alt text generator service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger channel factory.
   */
  public function __construct(
    AltTextGeneratorService $alt_text_generator,
    LoggerInterface  $logger
  ) {
    $this->altTextGenerator = $alt_text_generator;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('alttexting.alttext_generator'),
      $container->get('logger.factory')->get('alttexting')
    );
  }

  /**
   * Handles incoming webhook requests.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The HTTP request object.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The HTTP response object.
   */
  public function process(Request $request): Response {
    // Get query parameters.
    $media_id = $request->query->get('media_id', '');
    $langcode = $request->query->get('langcode', '');
    $image_field = $request->query->get('image_field', '');

    // Get request payload.
    $data = $request->request->all();
    $alt_text = $data['AltText'] ?? '';
    $error = $data['Error'] ?? '';

    // Log the webhook receipt.
    $this->logger->notice('Webhook received @media_id: AltText=@$alt_text Error=@error Langcode=@langcode, ImageField=@image_field', [
      '@media_id' => $media_id,
      '@alt_text' => $alt_text,
      '@error' => $error,
      '@langcode' => $langcode,
      '@image_field' => $image_field,
    ]);

    // Validate required parameters.
    if (empty($media_id) || empty($langcode) || empty($alt_text) || empty($image_field)) {
      return new Response('Missing required parameters', Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    try {
      $this->altTextGenerator->saveAltText($media_id, $langcode, $image_field, $alt_text);
      return new Response('OK', Response::HTTP_OK);
    }
    catch (Exception $e) {
      $this->logger->error('Error processing webhook: @message', [
        '@message' => $e->getMessage(),
        'exception' => $e,
      ]);
      return new Response('Error processing request', Response::HTTP_INTERNAL_SERVER_ERROR);
    }
  }
}
