<?php

namespace Drupal\alttexting\Plugin\QueueWorker;

use Drupal\Core\Annotation\QueueWorker;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Core\Queue\QueueWorkerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\alttexting\Service\AltTextGeneratorService;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;

/**
 * Processes media items for alt text generation.
 *
 * @QueueWorker(
 *   id = "alttexting_media_processor",
 *   title = @Translation("Alttext.ing Media Processor"),
 *   cron = {"time" = 60}
 * )
 */
class AltTextQueueWorker extends QueueWorkerBase implements QueueWorkerInterface, ContainerFactoryPluginInterface {

  /**
   * The logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The alt text generator service.
   *
   * @var \Drupal\alttexting\Service\AltTextGeneratorService
   */
  protected $altTextGenerator;

  /**
   * The maximum number of retry attempts for polling results.
   *
   * @var int
   */
  protected const MAX_RETRIES = 10;

  /**
   * The delay between retry attempts in seconds.
   *
   * @var int
   */
  protected const RETRY_DELAY = 3;

  /**
   * Constructs a new AltTextQueueWorker.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   * @param \Drupal\alttexting\Service\AltTextGeneratorService $alt_text_generator
   *   The alt text generator service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    LoggerInterface $logger,
    AltTextGeneratorService $alt_text_generator
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->logger = $logger;
    $this->altTextGenerator = $alt_text_generator;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('logger.factory')->get('alttexting'),
      $container->get('alttexting.alttext_generator')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data): void {
    if (!isset($data['media_id'], $data['fid'], $data['langcode'], $data['image_field'])) {
      $this->logger->error('Invalid queue item data: @data', ['@data' => print_r($data, TRUE)]);
      return;
    }

    $media_id = $data['media_id'];

    $this->logger->notice('Processing alt text for media @id (fid: @fid, lang: @langcode, field: @field)', [
      '@id' => $media_id,
      '@fid' => $data['fid'],
      '@langcode' => $data['langcode'],
      '@field' => $data['image_field']
      ]);

    try {
      // Generate alt text asynchronously.
      $result = $this->altTextGenerator->generateAsync(
        $data['fid'],
        $data['langcode'],
        $media_id,
        $data['image_field']
      );

      if (empty($result['id'])) {
        throw new \RuntimeException('Empty ID returned from alt text generation service');
      }

      $generated_data = $this->pollForResult($result['id']);

      if ($generated_data && $generated_data['hasAnswer'] && !empty($generated_data['altText'])) {
        $this->altTextGenerator->saveAltText(
          $media_id,
          $data['langcode'],
          $data['image_field'],
          $generated_data['altText']
        );
        $this->logger->info('Successfully generated and saved alt text for media @id', ['@id' => $media_id]);
      }
    }
    catch (\Exception $e) {
      $this->logger->error('Error processing alt text for media @id: @message', [
        '@id' => $media_id,
        '@message' => $e->getMessage(),
        'exception' => $e,
      ]);
      throw $e;
    }
  }

  /**
   * Polls for alt text generation result with retries.
   *
   * @param string $generation_id
   *   The alt text generation ID.
   *
   * @return array|null
   *   The result data if successful, or NULL if all retries failed.
   */
  protected function pollForResult(string $generation_id): ?array {
    $attempt = 0;

    while ($attempt < self::MAX_RETRIES) {
      try {
        $response = $this->altTextGenerator->tryGetResult($generation_id);

        if ($response->getStatusCode() !== Response::HTTP_OK) {
          $this->logger->warning('Received non-200 status code @code while polling for alt text', [
            '@code' => $response->getStatusCode(),
          ]);
          $attempt++;
          continue;
        }

        $content = json_decode($response->getContent(), TRUE);

        if (!is_array($content)) {
          throw new \RuntimeException('Invalid response format from alt text service');
        }

        if (!empty($content['altText']) && !isset($content['error'])) {
          return $content;
        }

        if (isset($content['error'])) {
          $this->logger->warning('Error in alt text generation: @error', [
            '@error' => $content['error'],
          ]);
          break;
        }
      }
      catch (\Exception $e) {
        $this->logger->error('Exception while polling for alt text: @message', [
          '@message' => $e->getMessage(),
          'exception' => $e,
        ]);
        // Continue to next attempt.
      }

      $attempt++;
      if ($attempt < self::MAX_RETRIES) {
        sleep(self::RETRY_DELAY);
      }
    }

    $this->logger->warning('Failed to get alt text after @attempt attempts', [
      '@attempt' => $attempt,
    ]);

    return NULL;
  }

}

