<?php

declare(strict_types=1);

namespace Drupal\alttexting\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\file\FileInterface;
use Drupal\image\Entity\ImageStyle;
use Drupal\image\ImageStyleInterface;
use Exception;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use Psr\Log\LoggerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Drupal\Core\Routing\UrlGeneratorInterface;

/**
 * Service for generating alt text using an external API.
 */
class AltTextGeneratorService {

  /**
   * The configuration object.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $config;

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\Client
   */
  protected Client $httpClient;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * The file URL generator.
   *
   * @var \Drupal\Core\File\FileUrlGeneratorInterface
   */
  protected FileUrlGeneratorInterface $fileUrlGenerator;

  /**
   * The URL generator.
   *
   * @var \Drupal\Core\Routing\UrlGeneratorInterface
   */
  protected UrlGeneratorInterface $urlGenerator;

  /**
   * Constructs a new AltTextGeneratorService object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \GuzzleHttp\Client $http_client
   *   The HTTP client.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\File\FileUrlGeneratorInterface $file_url_generator
   *   The file URL generator.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    Client $http_client,
    EntityTypeManagerInterface $entity_type_manager,
    LanguageManagerInterface $language_manager,
    LoggerChannelFactoryInterface $logger_factory,
    FileUrlGeneratorInterface $file_url_generator,
    UrlGeneratorInterface $urlGenerator
  ) {
    $this->config = $config_factory->get('alttexting.settings');
    $this->httpClient = $http_client;
    $this->entityTypeManager = $entity_type_manager;
    $this->languageManager = $language_manager;
    $this->logger = $logger_factory->get('alttexting');
    $this->fileUrlGenerator = $file_url_generator;
    $this->urlGenerator = $urlGenerator;
  }

  /**
   * Generates alt text for an image asynchronously.
   *
   * @param int $fid
   *   The file ID of the image to generate alt text for.
   * @param string $langcode
   *   The language code of the image to generate alt text for.
   * @param int|null $media_id
   *   The media ID of the image to generate alt text for.
   * @param string|null $image_field
   *   The image field of the image to generate alt text for.
   *
   * @return array
   *   The response from the API containing the operation ID.
   *
   * @throws \Exception When the API request fails.
   */
  public function generateAsync(int $fid, string $langcode, ?int $media_id = NULL, ?string $image_field = NULL): array {
    $api_url = $this->config->get('api_url');
    $langname = $this->languageManager->getLanguage($langcode)->getName();
    $image_url = $this->getImageUrlFromFid($fid);

    if (!$image_url) {
      $this->logger->error('Image not found: @image_url', ['@image_url' => $image_url]);
      return [];
    }

    $data = [
      'imageUrl' => $image_url,
      'lang' => $langname,
      'apiKey' => $this->config->get('api_key'),
    ];

    if ($this->config->get('autogenerate_on_save') && !$this->config->get('autogenerate_use_queue') && !empty($media_id) && !empty($image_field)) {
     // $data['webhook'] = "https://0156db7b3ce2.ngrok-free.app/alttexting/webhook?media_id=$media_id&langcode=$langcode&image_field=$image_field";
      $data['webhook'] = $this->urlGenerator->generateFromRoute(
        'alttexting.webhook',
        [],
        [
          'absolute' => TRUE,
          'query' => [
            'media_id' => $media_id,
            'langcode' => $langcode,
            'image_field' => $image_field,
          ],
        ]
      );
    }

    try {
      $response = $this->httpClient->post("$api_url/generate-async", [
        'json' => $data,
        'headers' => $this->getRequestHeaders([
          'Content-Type' => 'application/json',
        ]),
      ]);

      return json_decode($response->getBody()->getContents(), TRUE);
    }
    catch (GuzzleException $e) {
      throw new Exception('Failed to generate alt text: ' . $e->getMessage(), 0, $e);
    }
  }

  /**
   * Attempts to get the result of an async alt text generation.
   *
   * @param string $id
   *   The operation ID returned by generateAsync().
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response containing the result or error.
   *
   * @throws \GuzzleHttp\Exception\GuzzleException
   *   If the HTTP request fails.
   */
  /**
   * Tries to get the result of an async alt text generation.
   *
   * @param string $id
   *   The operation ID returned by generateAsync().
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response containing the result or error.
   */
  public function tryGetResult(string $id): JsonResponse {
    $api_url = $this->config->get('api_url');

    if (empty($api_url)) {
      return new JsonResponse(
        ['error' => 'API URL is not configured.'],
        JsonResponse::HTTP_INTERNAL_SERVER_ERROR
      );
    }

    try {
      $response = $this->httpClient->get(
        "{$api_url}/try-get-result",
        [
          'query' => ['id' => $id],
          'headers' => $this->getRequestHeaders(),
          'timeout' => 30,
        ]
      );

      $data = json_decode((string) $response->getBody(), TRUE);
      return new JsonResponse($data);
    }
    catch (GuzzleException $e) {
      $this->logger->error('Failed to get alt text result: @error', ['@error' => $e->getMessage()]);
      return new JsonResponse(
        ['error' => 'Failed to get result: ' . $e->getMessage()],
        JsonResponse::HTTP_INTERNAL_SERVER_ERROR
      );
    }
    catch (\JsonException $e) {
      $this->logger->error('Failed to decode alt text response: @error', ['@error' => $e->getMessage()]);
      return new JsonResponse(
        ['error' => 'Invalid response format from API'],
        JsonResponse::HTTP_INTERNAL_SERVER_ERROR
      );
    }
  }

  /**
   * Get the image URL from a file ID.
   *
   * @param int $fid
   *   The file ID.
   *
   * @return string|null
   *   The image URL, or NULL if the file couldn't be loaded.
   */
  /**
   * Gets the image URL from a file ID.
   *
   * @param int $fid
   *   The file ID.
   *
   * @return string|null
   *   The image URL, or NULL if the file couldn't be processed.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  public function getImageUrlFromFid(int $fid): ?string {
    $file_storage = $this->entityTypeManager->getStorage('file');
    $file = $file_storage->load($fid);

    if (!$file instanceof FileInterface) {
      return NULL;
    }

    $image_style = ImageStyle::load('alttexting');
    if (!$image_style instanceof ImageStyleInterface) {
      $this->logger->error('Image style "alttexting" is not configured.');
      return NULL;
    }

    // Get the URI of the styled image (resized).
    $styled_uri = $image_style->buildUri($file->getFileUri());

    // Generate the styled image if it doesn't exist yet.
    if (!file_exists($styled_uri)) {
      try {
        $image_style->createDerivative($file->getFileUri(), $styled_uri);
      }
      catch (\Exception $e) {
        $this->logger->error('Failed to create image derivative: @error', [
          '@error' => $e->getMessage(),
        ]);
        return NULL;
      }
    }

    if ($this->config->get('encode_image')) {
      $data = file_get_contents($styled_uri);
      if ($data === FALSE) {
        $this->logger->error('Failed to read image file: @uri', ['@uri' => $styled_uri]);
        return NULL;
      }

      $mime = $file->getMimeType();
      return 'data:' . $mime . ';base64,' . base64_encode($data);
    }

    // Convert the styled URI to an absolute URL.
    return $this->fileUrlGenerator->generateAbsoluteString($styled_uri);
  }

  /**
   * Gets the request headers with optional additional headers.
   *
   * @param array $headers
   *   Additional headers to include.
   *
   * @return array
   *   The complete headers array including any authorization headers.
   */
  private function getRequestHeaders(array $headers = []): array {
    $key = $this->config->get('authorization_key');
    $value = $this->config->get('authorization_value');

    if (!empty($key) && !empty($value)) {
      $headers[$key] = $value;
    }

    return $headers;
  }

  /**
   * Saves the generated alt text to a media entity.
   *
   * @param int $media_id
   *   The media entity ID.
   * @param string $langcode
   *   The language code.
   * @param string $image_field
   *   The image field name.
   * @param string $alt_text
   *   The alt text to save.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   * @throws \Drupal\Core\Entity\EntityStorageException
   * @throws \Drupal\Core\TypedData\Exception\MissingDataException
   */
  public function saveAltText(int $media_id, string $langcode, string $image_field, string $alt_text): void {
    $media_storage = $this->entityTypeManager->getStorage('media');
    $media = $media_storage->load($media_id);

    if (!$media) {
      $this->logger->error('Media not found: @media_id', ['@media_id' => $media_id]);
      return;
    }

    // Ensure the media has the requested translation.
    if ($media->hasTranslation($langcode)) {
      $media = $media->getTranslation($langcode);
    }

    // Validate the image field.
    if (!$media->hasField($image_field) || $media->get($image_field)->isEmpty()) {
      $this->logger->error('Image field not found or empty: @field in media @media_id', [
        '@field' => $image_field,
        '@media_id' => $media_id,
      ]);
      return;
    }

    $field = $media->get($image_field);
    $current_alt = $field->alt ?? '';

    // Skip if alt text is already set.
    if (!empty($current_alt)) {
      $this->logger->notice('Skipping media @id: alt text already set', [
        '@id' => $media_id,
      ]);
      return;
    }

    try {
      $field->first()->set('alt', $alt_text);
      $media->save();

      $this->logger->info('Alt text generated for media @id: @altText', [
        '@id' => $media_id,
        '@altText' => $alt_text,
      ]);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to save alt text for media @id: @error', [
        '@id' => $media_id,
        '@error' => $e->getMessage(),
      ]);
    }
  }

}
