<?php

namespace Drupal\alttexting\Controller;

use Exception;
use Psr\Log\LoggerInterface;
use Drupal\alttexting\Service\AltTextGeneratorService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;
use Drupal\Core\Controller\ControllerBase;

/**
 * Controller for handling alt text generation requests.
 */
class AltTextController extends ControllerBase {

  protected AltTextGeneratorService $altTextGenerator;
  protected LoggerInterface $logger;

  /**
   * Constructs a new AltTextController object.
   */
  public function __construct(
    AltTextGeneratorService $altTextGenerator,
    LoggerInterface $logger
  ) {
    $this->altTextGenerator = $altTextGenerator;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('alttexting.alttext_generator'),
      $container->get('logger.factory')->get('alttexting')
    );
  }
  /**
   * Generate alt text for an image asynchronously.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object containing 'fid' and 'langcode' parameters.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response containing the job ID or error message.
   */
  public function generateAsync(Request $request): JsonResponse {
    $fid = $request->request->get('fid');
    $langcode = $request->request->get('langcode');

    if (!$fid || !is_numeric($fid)) {
      throw new BadRequestHttpException('Invalid file ID provided.');
    }

    if (!$langcode || !is_string($langcode)) {
      throw new BadRequestHttpException('Invalid language code provided.');
    }

    try {
      $result = $this->altTextGenerator->generateAsync((int) $fid, $langcode);
      return new JsonResponse(['id' => $result['id']]);
    }
    catch (Exception $e) {
      $this->logger->error('Error generating alt text: @error', [
        '@error' => $e->getMessage(),
      ]);
      return new JsonResponse(
        ['error' => 'Failed to generate alt text. Please try again later.'],
        500
      );
    }
  }

  /**
   * Try to get the result of an alt text generation job.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object containing the job 'id' parameter.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The response containing the alt text or error message.
   *
   * @throws \GuzzleHttp\Exception\GuzzleException
   */
  public function tryGetResult(Request $request): JsonResponse {
    $id = $request->query->get('id');

    if (!$id || !is_string($id)) {
      throw new BadRequestHttpException('Invalid job ID provided.');
    }

    try {
      return $this->altTextGenerator->tryGetResult($id);
    }
    catch (Exception $e) {
      $this->logger->error('Error retrieving alt text result: @error', [
        '@error' => $e->getMessage(),
        '@job_id' => $id,
      ]);
      return new JsonResponse(
        ['error' => 'Failed to retrieve alt text result. Please try again.'],
        500
      );
    }
  }

}
