# API Plugins

A lightweight, programmer-focused Drupal module for integrating with external APIs through a plugin system.

## Core Philosophy

This module provides **only the essentials** for API integration:
- Plugin discovery and management
- HTTP request handling
- Authentication management
- Base classes for creating custom API plugins

**No UI, no forms, no fields** - just clean services for programmers.

## Installation

```bash
composer require drupal/api_plugins
drush en api_plugins
```

## Basic Usage

### Making API Requests

```php
$api_service = \Drupal::service('api_plugins.request');

// Call an API plugin
$result = $api_service->sendRequest('openai_chat_completions', [
  'prompt' => 'What is the capital city of Czechia?.',
  'model' => 'gpt-4o-mini',
  'temperature' => 0.0,
]);

print_r($result);
```

### Direct Plugin Usage

```php
// Get plugin manager
$plugin_manager = \Drupal::service('plugin.manager.api_endpoint');

// List all available plugins
$plugins = $plugin_manager->getDefinitions();

// Create plugin instance
$plugin = $plugin_manager->createInstance('openai_chat_completions');

// Prepare and send request
$payload = $plugin->preparePayload([
  'prompt' => 'Hello, world!',
  'model' => 'gpt-4o-mini',
]);
```

## Creating Custom Plugins

Create a plugin class in `src/Plugin/ApiPlugin/`:

```php
<?php

namespace Drupal\your_module\Plugin\ApiPlugin;

use Drupal\api_plugins\AiApiPluginBase;
use Drupal\api_plugins\ApiPluginInterface;

/**
 * @ApiPlugin(
 *   id = "my_api",
 *   label = @Translation("My API"),
 *   description = @Translation("Custom API integration"),
 *   type = "ai",
 *   endpointUrl = "https://api.example.com/v1/endpoint",
 *   defaultConfig = {
 *     "vendor" = "MyVendor",
 *     "model" = "default-model"
 *   }
 * )
 */
class MyApi extends AiApiPluginBase implements ApiPluginInterface {

  /**
   *
   */
  public function preparePayload(array $params = []): array {
    return [
      'model' => $this->getModel(),
      'input' => $params['prompt'] ?? '',
      'temperature' => $this->getTemperature(),
    ];
  }

  /**
   *
   */
  public function getHeaders(): array {
    return $this->getProviderHeaders();
  }

  /**
   *
   */
  public function formatResponse(array $result): string {
    return $result['output'] ?? '';
  }

  /**
   *
   */
  public function getAuthentication(string $plugin_id): string {
    return $this->getProviderAuthentication();
  }

  /**
   *
   */
  public function listModels(): array {
    return ['model-v1', 'model-v2'];
  }

}
```

## Included Plugins

- **anthropic_claude** - Anthropic Claude AI
- **openai_chat_completions** - OpenAI Chat API
- **openai_embeddings** - OpenAI Embeddings API


## Authentication

API keys are managed through the **Drupal Key module** for secure storage:

1. **Key module** (https://www.drupal.org/project/key) - Recommended
   - Secure storage (file, encrypted, environment)
   - UI-based management
   - Multiple storage backends

2. **Environment variables** - Fallback
   - Works if Key module is not installed
   - `OPENAI_API_KEY`, `ANTHROPIC_API_KEY`

**Setup:**
```bash
// Install Key module.
composer require drupal / key
drush en key

// Create keys at /admin/config/system/keys
// Configure at /admin/config/api_plugins/settings.
```

See **KEY_MODULE_INTEGRATION.md** for detailed setup instructions.

## Optional Submodules

- **api_plugins_feeds** - XML/RSS feed import functionality
- **api_plugins_ui** - Field types, widgets, and admin forms
- **api_plugins_mcp** - Model Context Protocol support

## Architecture

```
api_plugins /
├── src /
// Plugin contract.
│   ├── ApiPluginInterface . php
// Base plugin class.
│   ├── ApiPluginBase . php
// AI-specific base class.
│   ├── AiApiPluginBase . php
// Plugin manager.
│   ├── ApiPluginDiscovery . php
// HTTP request handler.
│   ├── ApiRequestService . php
// Auth management.
│   ├── ApiAuthenticationService . php
│   ├── Annotation /
// Plugin annotation.
│   │   └── ApiPlugin . php
│   └── Plugin / ApiPlugin /
│       ├── AnthropicClaude . php
│       ├── OpenAiChatCompletions . php
│       └── OpenAiEmbeddings . php
└── api_plugins . services . yml
```

## Services

- `api_plugins . request` - Main API request service
- `plugin . manager . api_endpoint` - Plugin discovery and manager
- `api_plugins . authentication` - Authentication service

## Requirements

- Drupal ^10 | ^11
- PHP 8.1+
- guzzlehttp/guzzle
