<?php

/**
 * @file
 * Hooks provided by the API Plugins module.
 */

/**
 * @addtogroup hooks
 * @{
 */

/**
 * Alter API plugin definitions.
 *
 * This hook allows modules to modify or add API plugin definitions
 * discovered by the plugin system.
 *
 * @param array &$definitions
 *   The array of plugin definitions, keyed by plugin ID.
 *
 * @see \Drupal\api_plugins\ApiPluginDiscovery
 */
function hook_api_plugins_plugin_definitions_alter(array &$definitions) {
  // Example: Modify an existing plugin definition.
  if (isset($definitions['openai_chat_completions'])) {
    $definitions['openai_chat_completions']['defaultConfig']['model'] = 'gpt-4';
  }

  // Example: Add a new plugin programmatically.
  $definitions['custom_api'] = [
    'id' => 'custom_api',
    'label' => t('Custom API'),
    'description' => t('Custom API integration'),
    'class' => 'Drupal\my_module\Plugin\ApiPlugin\CustomApi',
    'provider' => 'my_module',
    'type' => 'rest',
    'endpointUrl' => 'https://api.example.com',
    'defaultConfig' => [
      'vendor' => 'Custom',
      'timeout' => 30,
    ],
  ];
}

/**
 * Alter API request payload before sending.
 *
 * This hook allows modules to modify the payload that will be sent to
 * an API endpoint. This is invoked after the plugin's preparePayload()
 * method has been called.
 *
 * @param array &$payload
 *   The payload array that will be sent to the API, passed by reference.
 * @param array $context
 *   Context information:
 *   - 'plugin_id': The API plugin ID being used.
 *   - 'plugin': The API plugin instance.
 *   - 'params': The original parameters passed to sendRequest().
 *   - 'endpoint': The API endpoint URL.
 *
 * @see \Drupal\api_plugins\ApiRequestService::sendRequest()
 */
function hook_api_plugins_prepare_payload_alter(array &$payload, array $context) {
  // Example: Add custom headers or metadata for specific plugins.
  if ($context['plugin_id'] === 'openai_chat_completions') {
    // Add a custom system message to all OpenAI requests.
    if (isset($payload['messages'])) {
      array_unshift($payload['messages'], [
        'role' => 'system',
        'content' => 'You are a helpful assistant specialized in technical documentation.',
      ]);
    }
  }

  // Example: Modify temperature based on environment.
  if (isset($payload['temperature'])) {
    $config = \Drupal::config('my_module.settings');
    if ($config->get('override_temperature')) {
      $payload['temperature'] = $config->get('default_temperature');
    }
  }

  // Example: Add tracking metadata.
  $payload['metadata'] = [
    'site_id' => \Drupal::config('system.site')->get('uuid'),
    'timestamp' => time(),
    'user_id' => \Drupal::currentUser()->id(),
  ];

  // Example: Inject additional parameters for Anthropic Claude.
  if ($context['plugin_id'] === 'anthropic_claude') {
    $payload['max_tokens'] = $payload['max_tokens'] ?? 4096;
  }
}

/**
 * Define API key configuration fields for the API Plugins settings form.
 *
 * This hook allows modules (especially submodules) to register their own
 * API key configuration fields in the API Plugins settings form.
 *
 * Each module should return information about the API keys it needs,
 * and the form will automatically generate the appropriate fields with
 * status indicators.
 *
 * @return array
 *   An array of API key definitions, keyed by a unique machine name.
 *   Each definition should contain:
 *   - 'label': (string) The human-readable name of the API provider.
 *   - 'description': (string) Help text for the field.
 *   - 'config_key': (string) The configuration key to store the selected
 *     key ID (e.g., 'api_keys.openai_key').
 *   - 'weight': (int, optional) Display order weight. Default is 0.
 *
 * @see \Drupal\api_plugins\Form\ApiConfigForm::buildForm()
 */
function hook_api_plugins_api_key_info() {
  return [
    'openai' => [
      'label' => t('OpenAI API Key'),
      'description' => t('Select the key to use for OpenAI API authentication.'),
      'config_key' => 'api_keys.openai_key',
      'weight' => 0,
    ],
    'anthropic' => [
      'label' => t('Anthropic API Key'),
      'description' => t('Select the key to use for Anthropic Claude API authentication.'),
      'config_key' => 'api_keys.anthropic_key',
      'weight' => 10,
    ],
  ];
}

/**
 * Alter API key configuration information.
 *
 * This hook allows modules to modify API key definitions provided by
 * other modules before they are used to build the settings form.
 *
 * @param array &$key_info
 *   The array of API key definitions, passed by reference.
 *
 * @see hook_api_plugins_api_key_info()
 * @see \Drupal\api_plugins\Form\ApiConfigForm::buildForm()
 */
function hook_api_plugins_api_key_info_alter(array &$key_info) {
  // Example: Change the weight of OpenAI key field.
  if (isset($key_info['openai'])) {
    $key_info['openai']['weight'] = 100;
  }

  // Example: Add additional description.
  if (isset($key_info['anthropic'])) {
    $key_info['anthropic']['description'] .= ' ' . t('Required for Claude models.');
  }

  // Example: Remove a key field.
  unset($key_info['some_provider']);
}

/**
 * Define authentication configuration for API providers.
 *
 * This hook allows modules to register authentication configuration for
 * their API providers. The configuration specifies how to retrieve and
 * format authentication credentials.
 *
 * @return array
 *   An array of provider authentication configurations, keyed by provider
 *   machine name (lowercase). Each configuration should contain:
 *   - 'config_key': (string, optional) Configuration key for Key module
 *     key ID (e.g., 'api_keys.openai_key').
 *   - 'env_var': (string, optional) Environment variable name as fallback
 *     (e.g., 'OPENAI_API_KEY').
 *   - 'auth_type': (string, required) Authentication type:
 *     - 'bearer': Format as "Bearer {key}"
 *     - 'custom': Use key as-is
 *     - 'none': No authentication required
 *   - 'header_name': (string, optional) HTTP header name for authentication.
 *     Defaults to 'Authorization'.
 *
 * @see \Drupal\api_plugins\ApiAuthenticationService
 */
function hook_api_plugins_authentication_info() {
  return [
    'openai' => [
      'config_key' => 'api_keys.openai_key',
      'env_var' => 'OPENAI_API_KEY',
      'auth_type' => 'bearer',
      'header_name' => 'Authorization',
    ],
    'anthropic' => [
      'config_key' => 'api_keys.anthropic_key',
      'env_var' => 'ANTHROPIC_API_KEY',
      'auth_type' => 'custom',
      'header_name' => 'x-api-key',
    ],
  ];
}

/**
 * Alter authentication configuration for API providers.
 *
 * This hook allows modules to modify provider authentication configurations
 * registered by other modules.
 *
 * @param array &$auth_config
 *   The array of provider authentication configurations, passed by reference.
 *
 * @see hook_api_plugins_authentication_info()
 * @see \Drupal\api_plugins\ApiAuthenticationService
 */
function hook_api_plugins_authentication_info_alter(array &$auth_config) {
  // Example: Change authentication type for a provider.
  if (isset($auth_config['openai'])) {
    $auth_config['openai']['auth_type'] = 'custom';
  }

  // Example: Add environment variable fallback.
  if (isset($auth_config['anthropic'])) {
    $auth_config['anthropic']['env_var'] = 'CLAUDE_API_KEY';
  }

  // Example: Remove a provider configuration.
  unset($auth_config['some_provider']);
}

/**
 * @} End of "addtogroup hooks".
 */
