# API Plugins - Anthropic

Anthropic Claude AI integration for API Plugins.

## Features

- Claude 3 model family support (Opus, Sonnet, Haiku)
- Claude 3.5 Sonnet support
- System message handling
- Temperature and max tokens control
- Conversation history support
- Token replacement in prompts

## Installation

```bash
drush en api_plugins_anthropic
```

**Requires:** api_plugins (core module)

## Available Models

- `claude-3-opus-20240229` - Most powerful, best for complex tasks
- `claude-3-sonnet-20240229` - Balanced performance and speed
- `claude-3-haiku-20240307` - Fastest, most cost-effective (default)
- `claude-3-5-sonnet-20241022` - Latest Sonnet with improved capabilities

## Configuration

### Set API Key

```bash
# Via environment variable
export ANTHROPIC_API_KEY="sk-ant-..."

# Or configure via Drupal admin:
# /admin/config/services/api-plugins
```

### Via Key Module (Recommended)

1. Install the Key module
2. Create a new key at `/admin/config/system/keys`
3. Select it in API Plugins settings

## Usage

### Basic Example

```php
use Drupal\api_plugins\ApiRequestService;

$api_service = \Drupal::service('api_plugins.request');

$response = $api_service->sendRequest('anthropic_claude', [
  'prompt' => 'What is Drupal?',
  'model' => 'claude-3-haiku-20240307',
  'temperature' => 0.7,
  'max_tokens' => 1000,
]);

echo $response;
```

**Response:**
```
Drupal is a free and open-source content management system (CMS) written in PHP...
```

### Advanced Examples

#### 1. Using System Messages

```php
$response = $api_service->sendRequest('anthropic_claude', [
  'prompt' => 'Explain dependency injection',
  'system_message' => 'You are a Drupal expert. Explain concepts in the context of Drupal development.',
  'model' => 'claude-3-sonnet-20240229',
  'temperature' => 0.5,
  'max_tokens' => 2000,
]);
```

#### 2. Conversation History

```php
$response = $api_service->sendRequest('anthropic_claude', [
  'prompt' => 'What about services?',
  'system_message' => 'You are a Drupal expert.',
  'messages' => [
    ['role' => 'user', 'content' => 'What is dependency injection?'],
    ['role' => 'assistant', 'content' => 'Dependency injection is a design pattern...'],
  ],
  'model' => 'claude-3-opus-20240229',
]);
```

#### 3. Direct Plugin Usage

```php
$plugin = \Drupal::service('plugin.manager.api_endpoint')
  ->createInstance('anthropic_claude');

$plugin->setModel('claude-3-5-sonnet-20241022');
$plugin->setTemperature(0.8);
$plugin->setMaxTokens(4096);
$plugin->setSystemPrompt('You are a helpful Drupal developer assistant.');

$payload = $plugin->preparePayload([
  'prompt' => 'How do I create a custom block in Drupal 10?',
]);

// Send request using API service
$api_service = \Drupal::service('api_plugins.request');
$response = $api_service->sendRequest('anthropic_claude', [
  'prompt' => 'How do I create a custom block in Drupal 10?',
  'model' => 'claude-3-5-sonnet-20241022',
  'temperature' => 0.8,
  'max_tokens' => 4096,
]);
```

#### 4. Token Replacement in Prompts

```php
$response = $api_service->sendRequest('anthropic_claude', [
  'prompt' => 'Hello [current-user:name], explain [node:title]',
  'system_prompt' => 'Site: [site:name]',
  'token_data' => [
    'user' => \Drupal::currentUser(),
    'node' => $node,
  ],
  'model' => 'claude-3-haiku-20240307',
]);
```

#### 5. Code Generation

```php
$response = $api_service->sendRequest('anthropic_claude', [
  'prompt' => 'Create a Drupal 10 form that collects name and email',
  'system_message' => 'You are an expert Drupal developer. Generate clean, well-documented code following Drupal coding standards.',
  'model' => 'claude-3-opus-20240229',
  'temperature' => 0.3,
  'max_tokens' => 4000,
]);
```

#### 6. Content Analysis

```php
$response = $api_service->sendRequest('anthropic_claude', [
  'prompt' => "Analyze this content and suggest SEO improvements:\n\n" . $node->body->value,
  'system_message' => 'You are an SEO expert specializing in content optimization.',
  'model' => 'claude-3-sonnet-20240229',
  'temperature' => 0.5,
  'max_tokens' => 2000,
]);
```

#### 7. Batch Processing

```php
$nodes = \Drupal::entityTypeManager()
  ->getStorage('node')
  ->loadByProperties(['type' => 'article']);

foreach ($nodes as $node) {
  $summary = $api_service->sendRequest('anthropic_claude', [
    'prompt' => "Summarize this article in one sentence:\n\n" . $node->body->value,
    'model' => 'claude-3-haiku-20240307',
    'temperature' => 0.3,
    'max_tokens' => 100,
  ]);

  $node->set('field_ai_summary', $summary);
  $node->save();
}
```

## API Response Structure

Claude returns responses in this format:

```php
[
  'id' => 'msg_01...',
  'type' => 'message',
  'role' => 'assistant',
  'content' => [
    [
      'type' => 'text',
      'text' => 'The actual response text...',
    ],
  ],
  'model' => 'claude-3-haiku-20240307',
  'stop_reason' => 'end_turn',
  'usage' => [
    'input_tokens' => 15,
    'output_tokens' => 120,
  ],
]
```

The `formatResponse()` method extracts just the text content.

## Parameters

| Parameter | Type | Default | Description |
|-----------|------|---------|-------------|
| `prompt` | string | Required | The user message/prompt |
| `model` | string | `claude-3-haiku-20240307` | Claude model to use |
| `system_message` | string | null | System instructions for Claude |
| `system_prompt` | string | null | Alias for system_message |
| `temperature` | float | 0.0 | Randomness (0.0-2.0) |
| `max_tokens` | int | 1000 | Maximum tokens to generate |
| `messages` | array | [] | Conversation history |
| `token_data` | array | [] | Data for token replacement |
| `token_options` | array | [] | Options for token service |

## Model Selection Guide

### Claude 3 Opus
- **Best for:** Complex analysis, creative writing, detailed code generation
- **Speed:** Slower
- **Cost:** Higher
- **Use when:** Quality is paramount

### Claude 3.5 Sonnet
- **Best for:** Advanced coding, reasoning, and analysis
- **Speed:** Fast
- **Cost:** Moderate
- **Use when:** You need the latest capabilities with good speed

### Claude 3 Sonnet
- **Best for:** General tasks, content creation, analysis
- **Speed:** Medium
- **Cost:** Moderate
- **Use when:** Balanced performance is needed

### Claude 3 Haiku
- **Best for:** Simple tasks, summaries, quick responses
- **Speed:** Fastest
- **Cost:** Lowest
- **Use when:** Speed and cost matter more than complexity

## Error Handling

```php
try {
  $response = $api_service->sendRequest('anthropic_claude', [
    'prompt' => 'Hello',
    'model' => 'claude-3-haiku-20240307',
  ]);
} catch (\Exception $e) {
  \Drupal::logger('my_module')->error('Claude API error: @error', [
    '@error' => $e->getMessage(),
  ]);
}
```

## Common Errors

| Error | Cause | Solution |
|-------|-------|----------|
| "Anthropic API key isn't set or valid" | Missing API key | Set ANTHROPIC_API_KEY or configure in admin |
| "API request failed (HTTP 401)" | Invalid API key | Check your API key is correct |
| "API request failed (HTTP 429)" | Rate limit exceeded | Implement backoff/retry logic |
| "API request failed (HTTP 400)" | Invalid request | Check prompt and parameters |

## Authentication

The plugin uses the API Plugins authentication service with the following priority:

1. **Key module** - If configured in admin settings
2. **Environment variable** - `ANTHROPIC_API_KEY`
3. **Error** - Throws exception if neither is available

API keys should start with `sk-ant-` and can be obtained from:
https://console.anthropic.com/settings/keys

## Hooks

You can modify requests using hooks:

```php
/**
 * Implements hook_api_plugins_prepare_payload_alter().
 */
function mymodule_api_plugins_prepare_payload_alter(array &$payload, array $context) {
  if ($context['plugin_id'] === 'anthropic_claude') {
    // Force longer responses
    $payload['max_tokens'] = 4096;

    // Add safety settings
    $payload['metadata'] = [
      'user_id' => \Drupal::currentUser()->id(),
    ];
  }
}
```

## Best Practices

1. **Choose the right model:** Use Haiku for simple tasks, Opus for complex ones
2. **Set appropriate max_tokens:** Don't request more than needed
3. **Use system messages:** Guide Claude's behavior and expertise
4. **Handle errors gracefully:** Always wrap API calls in try-catch
5. **Cache responses:** For repeated prompts, cache the results
6. **Monitor usage:** Track token usage to manage costs
7. **Use environment variables:** Don't hardcode API keys

## Performance Tips

```php
// Bad: Multiple requests
foreach ($nodes as $node) {
  $result = $api_service->sendRequest('anthropic_claude', [
    'prompt' => "Summarize: {$node->getTitle()}",
  ]);
}

// Better: Batch in single request
$titles = array_map(fn($n) => $n->getTitle(), $nodes);
$result = $api_service->sendRequest('anthropic_claude', [
  'prompt' => "Summarize these articles:\n\n" . implode("\n", $titles),
  'max_tokens' => 2000,
]);
```

## Testing

```php
// Test API connection
drush php-eval "
  \$api = \\Drupal::service('api_plugins.request');
  \$response = \$api->sendRequest('anthropic_claude', [
    'prompt' => 'Say hello',
    'model' => 'claude-3-haiku-20240307',
    'max_tokens' => 10,
  ]);
  print \$response . PHP_EOL;
"
```

## Size

~125 lines of code

## Learn More

- [Anthropic Documentation](https://docs.anthropic.com/)
- [Claude Models Overview](https://docs.anthropic.com/claude/docs/models-overview)
- [API Reference](https://docs.anthropic.com/claude/reference/messages_post)
- [Prompt Engineering Guide](https://docs.anthropic.com/claude/docs/prompt-engineering)
