<?php

namespace Drupal\api_plugins_anthropic\Plugin\ApiPlugin;

use Drupal\api_plugins\AiApiPluginBase;
use Drupal\api_plugins\ApiPluginInterface;

/**
 * Provides Anthropic Claude API endpoint.
 *
 * @ApiPlugin(
 *   id = "anthropic_claude",
 *   label = @Translation("Anthropic Claude", context = "api_plugins"),
 *   description = @Translation("Anthropic Claude API endpoint.", context = "api_plugins"),
 *   type = "ai",
 *   endpointUrl = "https://api.anthropic.com/v1/messages",
 *   defaultConfig = {
 *     "vendor" = "Anthropic",
 *     "model" = "claude-3-haiku-20240307"
 *   }
 * )
 */
class AnthropicClaude extends AiApiPluginBase implements ApiPluginInterface {

  /**
   * {@inheritdoc}
   */
  protected string $model = 'claude-3-haiku-20240307';

  /**
   * {@inheritdoc}
   */
  protected float $temperature = 0.0;

  /**
   * {@inheritdoc}
   */
  public function preparePayload(array $params = []): array {
    $messages = $this->createMessages($params);
    if (empty($messages)) {
      throw new \InvalidArgumentException('At least one message is required for Claude API.');
    }

    $system_message = NULL;
    $filtered_messages = [];

    foreach ($messages as $message) {
      if ($message['role'] === 'system') {
        $system_message = $message['content'];
      }
      else {
        $filtered_messages[] = $message;
      }
    }

    $payload = [
      'model' => $this->getModel(),
      'messages' => $filtered_messages,
      'temperature' => $this->getTemperature(),
      'max_tokens' => $this->getMaxTokens() ?? 1000,
    ];

    if ($system_message) {
      $payload['system'] = $system_message;
    }

    return $payload;
  }

  /**
   * {@inheritdoc}
   *
   * Anthropic requires a special version header.
   */
  protected function getVendorSpecificHeaders(): array {
    return [
      'anthropic-version' => '2023-06-01',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function formatResponse(array $result): string {
    if (isset($result['content'][0]['text'])) {
      return $result['content'][0]['text'];
    }

    return '';
  }

  /**
   * {@inheritdoc}
   */
  public function listModels(): array {
    return [
      'claude-3-opus-20240229',
      'claude-3-sonnet-20240229',
      'claude-3-haiku-20240307',
      'claude-3-5-sonnet-20241022',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function validateResponse(array $response): bool {
    parent::validateResponse($response);

    $data = $response['data'] ?? $response;
    return isset($data['content'][0]['text']) && !empty($data['content'][0]['text']);
  }

}
