<?php

namespace Drupal\Tests\api_plugins_anthropic\Unit\Plugin\ApiPlugin;

use Drupal\api_plugins_anthropic\Plugin\ApiPlugin\AnthropicClaude;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests for AnthropicClaude plugin.
 */
#[CoversClass(AnthropicClaude::class)]
#[Group('api_plugins_anthropic')]
class AnthropicClaudeTest extends UnitTestCase {

  /**
   * The Anthropic Claude plugin.
   *
   * @var \Drupal\api_plugins_anthropic\Plugin\ApiPlugin\AnthropicClaude|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $plugin;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->plugin = $this->getMockBuilder(AnthropicClaude::class)
      ->disableOriginalConstructor()
      ->onlyMethods([
        'createMessages',
        't',
        'getModel',
        'getTemperature',
        'getMaxTokens',
        'getLogger',
        'getPluginId',
        'getProviderAuthentication',
      ])
      ->getMock();

    $this->plugin->method('t')
      ->willReturnCallback(function ($string) {
        return $string;
      });

    $this->plugin->method('getModel')
      ->willReturn('claude-3-haiku-20240307');

    $this->plugin->method('getTemperature')
      ->willReturn(0.0);

    $logger = $this->createMock('\Psr\Log\LoggerInterface');
    $this->plugin->method('getLogger')
      ->willReturn($logger);
  }

  /**
   * Tests preparePayload with system and user messages.
   */
  public function testPreparePayloadWithSystemAndUserMessages(): void {
    $messages = [
      ['role' => 'system', 'content' => 'You are helpful'],
      ['role' => 'user', 'content' => 'Hello'],
    ];

    $this->plugin->method('createMessages')
      ->willReturn($messages);

    $payload = $this->plugin->preparePayload(['prompt' => 'Hello']);

    $this->assertIsArray($payload);
    $this->assertArrayHasKey('system', $payload);
    $this->assertArrayHasKey('messages', $payload);
    $this->assertEquals('You are helpful', $payload['system']);

    // Only user message should be in messages array.
    $this->assertCount(1, $payload['messages']);
    $this->assertEquals('user', $payload['messages'][0]['role']);
  }

  /**
   * Tests preparePayload extracts system message correctly.
   */
  public function testPreparePayloadExtractsSystemMessage(): void {
    $messages = [
      ['role' => 'system', 'content' => 'System prompt'],
      ['role' => 'user', 'content' => 'User message'],
      ['role' => 'assistant', 'content' => 'Assistant response'],
      ['role' => 'user', 'content' => 'Follow-up'],
    ];

    $this->plugin->method('createMessages')
      ->willReturn($messages);

    $payload = $this->plugin->preparePayload([]);

    $this->assertEquals('System prompt', $payload['system']);
    $this->assertCount(3, $payload['messages']);
    foreach ($payload['messages'] as $message) {
      $this->assertNotEquals('system', $message['role']);
    }
  }

  /**
   * Tests preparePayload without system message.
   */
  public function testPreparePayloadWithoutSystemMessage(): void {
    $messages = [
      ['role' => 'user', 'content' => 'Hello'],
    ];

    $this->plugin->method('createMessages')
      ->willReturn($messages);

    $payload = $this->plugin->preparePayload(['prompt' => 'Hello']);

    $this->assertArrayNotHasKey('system', $payload);
    $this->assertArrayHasKey('messages', $payload);
    $this->assertCount(1, $payload['messages']);
  }

  /**
   * Tests preparePayload throws exception for empty messages.
   */
  public function testPreparePayloadThrowsExceptionForEmptyMessages(): void {
    $this->plugin->method('createMessages')
      ->willReturn([]);

    $this->expectException(\InvalidArgumentException::class);
    $this->expectExceptionMessage('At least one message is required');

    $this->plugin->preparePayload([]);
  }

  /**
   * Tests preparePayload includes required fields.
   */
  public function testPreparePayloadIncludesRequiredFields(): void {
    $messages = [['role' => 'user', 'content' => 'Test']];

    $this->plugin->method('createMessages')
      ->willReturn($messages);

    $payload = $this->plugin->preparePayload(['prompt' => 'Test']);

    $this->assertArrayHasKey('model', $payload);
    $this->assertArrayHasKey('messages', $payload);
    $this->assertArrayHasKey('temperature', $payload);
    $this->assertArrayHasKey('max_tokens', $payload);
  }

  /**
   * Tests preparePayload uses default max_tokens.
   */
  public function testPreparePayloadUsesDefaultMaxTokens(): void {
    $messages = [['role' => 'user', 'content' => 'Test']];

    $this->plugin->method('createMessages')
      ->willReturn($messages);

    $this->plugin->method('getMaxTokens')
      ->willReturn(NULL);

    $payload = $this->plugin->preparePayload(['prompt' => 'Test']);

    $this->assertEquals(1000, $payload['max_tokens']);
  }

  /**
   * Tests preparePayload uses custom max_tokens.
   */
  public function testPreparePayloadUsesCustomMaxTokens(): void {
    $messages = [['role' => 'user', 'content' => 'Test']];

    $this->plugin->method('createMessages')
      ->willReturn($messages);

    $this->plugin->method('getMaxTokens')
      ->willReturn(2000);

    $payload = $this->plugin->preparePayload(['prompt' => 'Test']);

    $this->assertEquals(2000, $payload['max_tokens']);
  }

  /**
   * Tests getVendorSpecificHeaders returns anthropic-version header.
   */
  public function testGetVendorSpecificHeaders(): void {
    $reflection = new \ReflectionClass($this->plugin);
    $method = $reflection->getMethod('getVendorSpecificHeaders');
    $method->setAccessible(TRUE);

    $headers = $method->invoke($this->plugin);

    $this->assertIsArray($headers);
    $this->assertArrayHasKey('anthropic-version', $headers);
    $this->assertEquals('2023-06-01', $headers['anthropic-version']);
  }

  /**
   * Tests formatResponse extracts text content.
   */
  public function testFormatResponseExtractsContent(): void {
    $result = [
      'id' => 'msg_123',
      'type' => 'message',
      'role' => 'assistant',
      'content' => [
        [
          'type' => 'text',
          'text' => 'Hello! How can I assist you today?',
        ],
      ],
      'model' => 'claude-3-haiku-20240307',
      'usage' => [
        'input_tokens' => 10,
        'output_tokens' => 8,
      ],
    ];

    $formatted = $this->plugin->formatResponse($result);

    $this->assertEquals('Hello! How can I assist you today?', $formatted);
  }

  /**
   * Tests formatResponse returns empty string for missing content.
   */
  public function testFormatResponseReturnsEmptyForMissingContent(): void {
    $result = ['content' => []];

    $formatted = $this->plugin->formatResponse($result);

    $this->assertEquals('', $formatted);
  }

  /**
   * Tests listModels returns available Claude models.
   */
  public function testListModels(): void {
    $models = $this->plugin->listModels();

    $this->assertIsArray($models);
    $this->assertNotEmpty($models);
    $this->assertContains('claude-3-opus-20240229', $models);
    $this->assertContains('claude-3-sonnet-20240229', $models);
    $this->assertContains('claude-3-haiku-20240307', $models);
    $this->assertContains('claude-3-5-sonnet-20241022', $models);
  }

  /**
   * Tests validateResponse with valid content.
   */
  public function testValidateResponseWithValidContent(): void {
    $response = [
      'data' => [
        'content' => [
          ['text' => 'Valid response'],
        ],
      ],
    ];

    $result = $this->plugin->validateResponse($response);

    $this->assertTrue($result);
  }

  /**
   * Tests validateResponse with empty text.
   */
  public function testValidateResponseWithEmptyText(): void {
    $response = [
      'data' => [
        'content' => [
          ['text' => ''],
        ],
      ],
    ];

    $result = $this->plugin->validateResponse($response);

    $this->assertFalse($result);
  }

  /**
   * Tests validateResponse with missing content.
   */
  public function testValidateResponseWithMissingContent(): void {
    $response = ['data' => []];

    $result = $this->plugin->validateResponse($response);

    $this->assertFalse($result);
  }

  /**
   * Tests preparePayload with multiple system messages (uses last one).
   */
  public function testPreparePayloadWithMultipleSystemMessages(): void {
    $messages = [
      ['role' => 'system', 'content' => 'First system'],
      ['role' => 'system', 'content' => 'Second system'],
      ['role' => 'user', 'content' => 'User message'],
    ];

    $this->plugin->method('createMessages')
      ->willReturn($messages);

    $payload = $this->plugin->preparePayload([]);
    $this->assertEquals('Second system', $payload['system']);
    $this->assertCount(1, $payload['messages']);
  }

  /**
   * Tests default model is claude-3-haiku.
   */
  public function testDefaultModelIsClaudeHaiku(): void {
    $reflection = new \ReflectionClass(AnthropicClaude::class);
    $property = $reflection->getProperty('model');
    $property->setAccessible(TRUE);

    $plugin = $reflection->newInstanceWithoutConstructor();
    $model = $property->getValue($plugin);

    $this->assertEquals('claude-3-haiku-20240307', $model);
  }

  /**
   * Tests default temperature is 0.0.
   */
  public function testDefaultTemperatureIsZero(): void {
    $reflection = new \ReflectionClass(AnthropicClaude::class);
    $property = $reflection->getProperty('temperature');
    $property->setAccessible(TRUE);

    $plugin = $reflection->newInstanceWithoutConstructor();
    $temperature = $property->getValue($plugin);

    $this->assertEquals(0.0, $temperature);
  }

}
