# API Plugins - MCP

Model Context Protocol (MCP) support for API Plugins.

## Features

- MCP protocol implementation
- Session management
- Tool discovery and execution
- Base classes for MCP servers

## Installation

```bash
drush en api_plugins_mcp
```

**Requires:** api_plugins (core module)

## Usage

### Apify MCP Server Examples

The Apify MCP Server plugin provides access to all major Apify tool categories:
- **actors** - Actor discovery and management tools
- **docs** - Documentation search tools
- **runs** - Actor run information tools
- **storage** - Dataset and key-value store tools
- **Specific actors** - Like `apify/rag-web-browser` and `compass/crawler-google-places`

#### 1. List Available Tools

```php
$plugin_manager = \Drupal::service('plugin.manager.api_endpoint');
$plugin = $plugin_manager->createInstance('apify_mcp_server');
$response = $plugin->sendRequest([
  'method' => 'tools/list',
  'id' => 1,
]);

print_r($response);
```

#### 2. Search and Discover Actors

The `actors` category enables dynamic tool discovery, allowing you to find and use any actor from the Apify Store.

```php
$plugin_manager = \Drupal::service('plugin.manager.api_endpoint');
$plugin = $plugin_manager->createInstance('apify_mcp_server');

// Search for actors
$response = $plugin->sendRequest([
  'method' => 'tools/call',
  'id' => 2,
  'tool_name' => 'search-actors',
  'arguments' => [
    'query' => 'instagram scraper',
  ],
]);

print_r($response);
```

#### 3. Call a Tool - Run an Actor

```php
$plugin_manager = \Drupal::service('plugin.manager.api_endpoint');
$plugin = $plugin_manager->createInstance('apify_mcp_server');

$response = $plugin->sendRequest([
  'method' => 'tools/call',
  'id' => 3,
  'tool_name' => 'call-actor',
  'arguments' => [
    'step' => 'call',
    'actor' => 'apify/web-scraper',
    'input' => [
      'startUrls' => [
        ['url' => 'https://ludekkvapil.cz'],
      ],
      'pageFunction' => 'async function pageFunction(context) {
        return {
          url: context.request.url,
          title: context.jQuery("title").text(),
          heading: context.jQuery("h1").first().text(),
          timestamp: new Date().toISOString()
        };
      }',
      'proxyConfiguration' => [
        'useApifyProxy' => true,
      ],
    ],
  ],
]);

print_r($response);
```

#### 4. Web Scraping with RAG Browser

**Note:** The `apify/rag-web-browser` tool requires a `query` parameter for search-based web browsing.

```php
$plugin_manager = \Drupal::service('plugin.manager.api_endpoint');
$plugin = $plugin_manager->createInstance('apify_mcp_server');

$response = $plugin->sendRequest([
  'method' => 'tools/call',
  'id' => 4,
  'tool_name' => 'apify/rag-web-browser',
  'arguments' => [
    'query' => 'Drupal content management features',  // Required parameter
    'url' => 'https://www.drupal.org',
    'maxCrawlDepth' => 1,
    'crawlerType' => 'playwright:chromium',
  ],
]);

print_r($response);
```

#### 5. Crawl Google Places Using call-actor

Since `compass/crawler-google-places` is not available as a dedicated tool, use the generic `call-actor` tool with the mandatory two-step workflow:

**Step 1: Get Actor Info**

```php
$plugin_manager = \Drupal::service('plugin.manager.api_endpoint');
$plugin = $plugin_manager->createInstance('apify_mcp_server');

// First, get the actor's input schema
$info_response = $plugin->sendRequest([
  'method' => 'tools/call',
  'id' => 5,
  'tool_name' => 'call-actor',
  'arguments' => [
    'actor' => 'compass/crawler-google-places',
    'step' => 'info',  // Get actor details and input schema first
  ],
]);

print_r($info_response);
```

**Step 2: Call the Actor**

```php
// After reviewing the input schema from step 1, call the actor with proper input
$response = $plugin->sendRequest([
  'method' => 'tools/call',
  'id' => 6,
  'tool_name' => 'call-actor',
  'arguments' => [
    'actor' => 'compass/crawler-google-places',
    'step' => 'call',
    'input' => [
      // Use the exact parameters from the input schema
      'searchStringsArray' => ['restaurants in Prague'],
      'maxCrawledPlacesPerSearch' => 20,
    ],
  ],
]);

print_r($response);
```

#### 6. Access Actor Run Information

```php
$plugin_manager = \Drupal::service('plugin.manager.api_endpoint');
$plugin = $plugin_manager->createInstance('apify_mcp_server');

$response = $plugin->sendRequest([
  'method' => 'tools/call',
  'id' => 6,
  'tool_name' => 'get-actor-run',
  'arguments' => [
    'runId' => 'your_run_id_here',
  ],
]);

print_r($response);
```

#### 7. Search Apify Documentation

```php
$plugin_manager = \Drupal::service('plugin.manager.api_endpoint');
$plugin = $plugin_manager->createInstance('apify_mcp_server');

$response = $plugin->sendRequest([
  'method' => 'tools/call',
  'id' => 7,
  'tool_name' => 'search-apify-docs',
  'arguments' => [
    'query' => 'web scraping best practices',
  ],
]);

print_r($response);
```

#### 8. Access Dataset Items

```php
$plugin_manager = \Drupal::service('plugin.manager.api_endpoint');
$plugin = $plugin_manager->createInstance('apify_mcp_server');

$response = $plugin->sendRequest([
  'method' => 'tools/call',
  'id' => 8,
  'tool_name' => 'get-dataset-items',
  'arguments' => [
    'datasetId' => 'your_dataset_id_here',
    'limit' => 100,
  ],
]);

print_r($response);
```

### Configuring Available Tools

You can customize which tools are available by modifying the `$apifyTools` property in the plugin or by using the `setApifyTools()` method:

```php
$plugin_manager = \Drupal::service('plugin.manager.api_endpoint');
$plugin = $plugin_manager->createInstance('apify_mcp_server');

// Add specific actors to the available tools
$plugin->setApifyTools([
  'actors',
  'docs',
  'runs',
  'storage',
  'apify/rag-web-browser',
  'apify/instagram-scraper',
  'apify/google-search-scraper',
]);

$response = $plugin->sendRequest([
  'method' => 'tools/list',
  'id' => 1,
]);

print_r($response);
```

## Understanding Dedicated Tools vs. call-actor

### Dedicated Actor Tools

Some actors have **dedicated MCP tools** that provide a simplified interface:
- `apify/rag-web-browser` → `apify-slash-rag-web-browser` tool
- These tools have their own input schemas and are listed directly in `tools/list`
- You can call them directly without the two-step workflow

### Generic call-actor Tool

For actors **without dedicated tools**, use the generic `call-actor` tool:
- **Mandatory two-step workflow:**
  1. **Step 1 (info):** Get actor details and input schema
  2. **Step 2 (call):** Run the actor with proper input
- Examples: `compass/crawler-google-places`, `apify/instagram-scraper`, etc.

### How to Check

Run `tools/list` to see which actors have dedicated tools:

```php
$response = $plugin->sendRequest([
  'method' => 'tools/list',
  'id' => 1,
]);

// Look for tools with names like "apify-slash-actor-name"
// or "compass-slash-actor-name"
foreach ($response['tools'] as $tool) {
  echo $tool['name'] . "\n";
}
```

## Troubleshooting

### Understanding Tool Input Schemas

Each Apify actor has its own input schema. To find the correct parameters for a tool:

1. **List all available tools** to see their input schemas:

```php
$plugin_manager = \Drupal::service('plugin.manager.api_endpoint');
$plugin = $plugin_manager->createInstance('apify_mcp_server');

$response = $plugin->sendRequest([
  'method' => 'tools/list',
  'id' => 1,
]);

// Each tool in the response will have an 'inputSchema' property
// that describes the required and optional parameters
foreach ($response['tools'] as $tool) {
  echo "Tool: " . $tool['name'] . "\n";
  echo "Input Schema:\n";
  print_r($tool['inputSchema']);
  echo "\n\n";
}
```

2. **Use the search-actors tool** to find actors and their documentation:

```php
$response = $plugin->sendRequest([
  'method' => 'tools/call',
  'id' => 2,
  'tool_name' => 'search-actors',
  'arguments' => [
    'query' => 'google places',
  ],
]);

print_r($response);
```

3. **Check the Apify documentation** for specific actors at `https://apify.com/[actor-id]`

### Common Issues

**Issue:** Actor completes but returns 0 items
- **Solution:** Check the input schema using `tools/list` - you may be using incorrect parameter names or missing required fields

**Issue:** "Invalid arguments" error
- **Solution:** The tool requires specific parameters. Use `tools/list` to see the `inputSchema` for the tool and ensure all required properties are provided

**Issue:** Tool not found
- **Solution:** Make sure the tool is included in `$apifyTools` array and that you're using the correct tool name format (e.g., `apify/actor-name` or `compass/crawler-name`)
