<?php

namespace Drupal\api_plugins_mcp;

use Drupal\api_plugins\ApiPluginBase;

/**
 * Abstract base class for MCP API plugins.
 *
 * Provides common functionality shared between MCP (Model Context Protocol)
 * services. This includes tool management, capability discovery, and
 * standardized MCP methods like callTool(), listTools(), and getDocs().
 *
 * @package Drupal\api_plugins
 */
abstract class McpApiPluginBase extends ApiPluginBase {

  /**
   * The MCP server endpoint URL.
   *
   * @var string
   */
  protected $mcpServerUrl = '';

  /**
   * Available MCP tools/capabilities.
   *
   * @var array<string, mixed>
   */
  protected $availableTools = [];

  /**
   * MCP protocol version.
   *
   * @var string
   */
  protected $protocolVersion = '2024-11-05';

  /**
   * Client capabilities.
   *
   * @var array<string, mixed>
   */
  protected $clientCapabilities = [
    'roots' => [
      'listChanged' => TRUE,
    ],
    'sampling' => [],
  ];

  /**
   * Server capabilities (populated after connection).
   *
   * @var array<string, mixed>
   */
  protected $serverCapabilities = [];

  /**
   * MCP session initialized flag.
   *
   * @var bool
   */
  protected $sessionInitialized = FALSE;

  /**
   * Get MCP server URL.
   *
   * @return string
   *   The MCP server URL.
   */
  public function getMcpServerUrl(): string {
    return $this->mcpServerUrl;
  }

  /**
   * Set MCP server URL.
   *
   * @param string $url
   *   The MCP server URL.
   *
   * @return $this
   *   Returns self for method chaining.
   *
   * @throws \InvalidArgumentException
   *   If URL is invalid or points to private/reserved IP ranges.
   */
  public function setMcpServerUrl(string $url): static {
    $this->validateUrl($url);

    $this->mcpServerUrl = $url;
    return $this;
  }

  /**
   * Validates a URL for security and format.
   *
   * Prevents SSRF attacks by:
   * - Validating URL structure
   * - Restricting to HTTP/HTTPS schemes
   * - Blocking private and reserved IP ranges
   * - Checking all resolved IPs to prevent DNS rebinding attacks.
   *
   * @param string $url
   *   The URL to validate.
   *
   * @throws \InvalidArgumentException
   *   If URL is invalid or insecure.
   */
  protected function validateUrl(string $url): void {
    $parsed = parse_url($url);
    if (!$parsed || !isset($parsed['scheme'], $parsed['host'])) {
      throw new \InvalidArgumentException('Invalid URL format.');
    }

    $allowed_schemes = ['https', 'http'];
    if (!in_array($parsed['scheme'], $allowed_schemes, TRUE)) {
      throw new \InvalidArgumentException('Only HTTP and HTTPS schemes are allowed.');
    }

    $host = $parsed['host'];

    // Check for forbidden localhost variants.
    $forbidden_hosts = ['localhost', '127.0.0.1', '::1', '0.0.0.0', '0000:0000:0000:0000:0000:0000:0000:0001'];
    if (in_array(strtolower($host), $forbidden_hosts, TRUE)) {
      throw new \InvalidArgumentException('Cannot access localhost.');
    }

    // If direct IP address, validate it.
    if (filter_var($host, FILTER_VALIDATE_IP)) {
      if (filter_var($host, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) === FALSE) {
        throw new \InvalidArgumentException('Cannot access private or reserved IP addresses.');
      }
    }
    else {
      // For hostnames, resolve ALL IPs and validate each one.
      // This prevents DNS rebinding attacks.
      $ips = @gethostbynamel($host);
      if ($ips === FALSE || empty($ips)) {
        // Fallback to single lookup.
        $ip = @gethostbyname($host);
        if ($ip !== $host) {
          $ips = [$ip];
        }
        else {
          throw new \InvalidArgumentException('Cannot resolve hostname.');
        }
      }

      // Validate each resolved IP address.
      foreach ($ips as $ip) {
        if (filter_var($ip, FILTER_VALIDATE_IP)) {
          if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) === FALSE) {
            throw new \InvalidArgumentException('Cannot access hosts that resolve to private or reserved IP addresses.');
          }
        }
      }
    }

    // Sanitize URL for logging (remove sensitive query params).
    $safe_url = $this->sanitizeUrlForLogging($url);
    $this->getLogger('api_plugins')->info(
      'MCP server URL validated: @url',
      ['@url' => $safe_url]
    );
  }

  /**
   * Sanitizes URL for logging by removing sensitive query parameters.
   *
   * @param string $url
   *   The URL to sanitize.
   *
   * @return string
   *   The sanitized URL.
   */
  protected function sanitizeUrlForLogging(string $url): string {
    $parsed = parse_url($url);
    if (isset($parsed['query'])) {
      $parsed['query'] = '***REDACTED***';
    }
    if (isset($parsed['user']) || isset($parsed['pass'])) {
      unset($parsed['user'], $parsed['pass']);
    }

    // Rebuild URL.
    $sanitized = $parsed['scheme'] . '://';
    if (isset($parsed['host'])) {
      $sanitized .= $parsed['host'];
    }
    if (isset($parsed['port'])) {
      $sanitized .= ':' . $parsed['port'];
    }
    if (isset($parsed['path'])) {
      $sanitized .= $parsed['path'];
    }
    if (isset($parsed['query'])) {
      $sanitized .= '?' . $parsed['query'];
    }

    return $sanitized;
  }

  /**
   * Get protocol version.
   *
   * @return string
   *   The MCP protocol version.
   */
  public function getProtocolVersion(): string {
    return $this->protocolVersion;
  }

  /**
   * Set protocol version.
   *
   * @param string $version
   *   The protocol version.
   *
   * @return $this
   *   Returns self for method chaining.
   */
  public function setProtocolVersion(string $version): static {
    $this->protocolVersion = $version;
    return $this;
  }

  /**
   * Get client capabilities.
   *
   * @return array
   *   Client capabilities array.
   */
  public function getClientCapabilities(): array {
    return $this->clientCapabilities;
  }

  /**
   * Set client capabilities.
   *
   * @param array $capabilities
   *   Client capabilities.
   *
   * @return $this
   *   Returns self for method chaining.
   */
  public function setClientCapabilities(array $capabilities): static {
    $this->clientCapabilities = $capabilities;
    return $this;
  }

  /**
   * Get server capabilities.
   *
   * @return array
   *   Server capabilities array.
   */
  public function getServerCapabilities(): array {
    return $this->serverCapabilities;
  }

  /**
   * Check if session is initialized.
   *
   * @return bool
   *   TRUE if session is initialized.
   */
  public function isSessionInitialized(): bool {
    return $this->sessionInitialized;
  }

  /**
   * Initialize MCP session.
   *
   * @return array
   *   Initialization response from server.
   *
   * @throws \Exception
   *   If initialization fails.
   */
  public function initializeSession(): array {
    $payload = [
      'jsonrpc' => '2.0',
      'id' => 1,
      'method' => 'initialize',
      'params' => [
        'protocolVersion' => $this->protocolVersion,
        'capabilities' => $this->clientCapabilities,
        'clientInfo' => [
          'name' => 'Drupal API Plugins',
          'version' => '1.0.0',
        ],
      ],
    ];

    try {
      $response = $this->sendMcpRequest($payload);

      if (isset($response['result']['capabilities'])) {
        $this->serverCapabilities = $response['result']['capabilities'];
        $this->sessionInitialized = TRUE;
      }

      return $response;
    }
    catch (\Exception $e) {
      throw new \Exception('MCP session initialization failed: ' . $e->getMessage());
    }
  }

  /**
   * List available tools from MCP server.
   *
   * @return array
   *   Array of available tools with their schemas.
   *
   * @throws \Exception
   *   If listing tools fails.
   */
  public function listTools(): array {
    if (!$this->sessionInitialized) {
      $this->initializeSession();
    }

    $payload = [
      'jsonrpc' => '2.0',
      'id' => 2,
      'method' => 'tools/list',
      'params' => [],
    ];

    try {
      $response = $this->sendMcpRequest($payload);

      if (isset($response['result']['tools'])) {
        $this->availableTools = $response['result']['tools'];
        return $this->availableTools;
      }

      return [];
    }
    catch (\Exception $e) {
      throw new \Exception('Failed to list MCP tools: ' . $e->getMessage());
    }
  }

  /**
   * Call a specific MCP tool.
   *
   * @param string $toolName
   *   The name of the tool to call.
   * @param array $arguments
   *   Arguments to pass to the tool.
   *
   * @return array
   *   Tool execution result.
   *
   * @throws \Exception
   *   If tool call fails.
   */
  public function callTool(string $toolName, array $arguments = []): array {
    if (!$this->sessionInitialized) {
      $this->initializeSession();
    }

    $payload = [
      'jsonrpc' => '2.0',
      'id' => $this->generateSecureRequestId(),
      'method' => 'tools/call',
      'params' => [
        'name' => $toolName,
        'arguments' => $arguments,
      ],
    ];

    try {
      $response = $this->sendMcpRequest($payload);

      if (isset($response['result'])) {
        return $response['result'];
      }

      if (isset($response['error'])) {
        throw new \Exception('MCP tool error: ' . ($response['error']['message'] ?? 'Unknown error'));
      }

      return [];
    }
    catch (\Exception $e) {
      throw new \Exception('Failed to call MCP tool "' . $toolName . '": ' . $e->getMessage());
    }
  }

  /**
   * Get documentation for available tools.
   *
   * @param string|null $toolName
   *   Specific tool name, or null for all tools.
   *
   * @return array
   *   Tool documentation.
   */
  public function getDocs(?string $toolName = NULL): array {
    $tools = $this->listTools();

    if ($toolName !== NULL) {
      foreach ($tools as $tool) {
        if ($tool['name'] === $toolName) {
          return [
            'name' => $tool['name'],
            'description' => $tool['description'] ?? '',
            'inputSchema' => $tool['inputSchema'] ?? [],
          ];
        }
      }
      return [];
    }

    // Return documentation for all tools.
    $docs = [];
    foreach ($tools as $tool) {
      $docs[] = [
        'name' => $tool['name'],
        'description' => $tool['description'] ?? '',
        'inputSchema' => $tool['inputSchema'] ?? [],
      ];
    }

    return $docs;
  }

  /**
   * Send MCP request to server.
   *
   * @param array $payload
   *   The MCP request payload.
   *
   * @return array
   *   The MCP response.
   *
   * @throws \Exception
   *   If the request fails.
   */
  abstract protected function sendMcpRequest(array $payload): array;

  /**
   * Get available tools (cached).
   *
   * @return array
   *   Array of available tools.
   */
  public function getAvailableTools(): array {
    if (empty($this->availableTools)) {
      $this->listTools();
    }
    return $this->availableTools;
  }

  /**
   * Check if a specific tool is available.
   *
   * @param string $toolName
   *   The tool name to check.
   *
   * @return bool
   *   TRUE if tool is available.
   */
  public function isToolAvailable(string $toolName): bool {
    $tools = $this->getAvailableTools();
    foreach ($tools as $tool) {
      if ($tool['name'] === $toolName) {
        return TRUE;
      }
    }
    return FALSE;
  }

  /**
   * Get MCP-specific configuration parameters.
   *
   * Returns common MCP parameters that can be used in payload preparation.
   *
   * @return array
   *   Array of MCP configuration parameters.
   */
  protected function getMcpParameters(): array {
    return [
      'protocolVersion' => $this->getProtocolVersion(),
      'serverUrl' => $this->getMcpServerUrl(),
      'capabilities' => $this->getClientCapabilities(),
      'sessionInitialized' => $this->isSessionInitialized(),
    ];
  }

  /**
   * Set multiple MCP parameters at once.
   *
   * @param array $params
   *   Array of parameters to set.
   *
   * @return $this
   *   Returns self for method chaining.
   */
  public function setMcpParameters(array $params): static {
    if (isset($params['server_url'])) {
      $this->setMcpServerUrl($params['server_url']);
    }

    if (isset($params['protocol_version'])) {
      $this->setProtocolVersion($params['protocol_version']);
    }

    if (isset($params['client_capabilities'])) {
      $this->setClientCapabilities($params['client_capabilities']);
    }

    return $this;
  }

  /**
   * Get all current MCP parameters.
   *
   * @return array
   *   Array of all current MCP parameters.
   */
  public function getAllMcpParameters(): array {
    return [
      'server_url' => $this->getMcpServerUrl(),
      'protocol_version' => $this->getProtocolVersion(),
      'client_capabilities' => $this->getClientCapabilities(),
      'server_capabilities' => $this->getServerCapabilities(),
      'session_initialized' => $this->isSessionInitialized(),
      'available_tools' => $this->getAvailableTools(),
    ];
  }

  /**
   * Send an MCP request with simplified interface.
   *
   * This provides a convenient way to make MCP requests directly on the plugin
   * instance, similar to the ApiRequestService pattern but without needing
   * the service layer.
   *
   * @param array $params
   *   Request parameters including:
   *   - method: The MCP method (initialize, tools/list, tools/call, etc.)
   *   - id: Request ID (optional, will auto-generate if not provided)
   *   - tool_name: For tools/call, the name of the tool
   *   - arguments: For tools/call, the tool arguments
   *   - Any other method-specific parameters.
   *
   * @return mixed
   *   The formatted response from the MCP server.
   *
   * @throws \Exception
   *   If the request fails.
   */
  public function sendRequest(array $params = []) {
    $this->prepareForRequest($params);
    $payload = $this->preparePayload($params);
    $response = $this->sendMcpRequest($payload);
    return $this->formatResponse($response);
  }

  /**
   * Prepare the plugin before a request is made.
   *
   * This is invoked by the universal request service prior to calling
   * {@see preparePayload()}. For MCP plugins this ensures the MCP session
   * is initialized and allows passing initialization parameters via
   * the service-level params array (e.g. 'capabilities' or 'clientInfo').
   *
   * @param array $params
   *   Incoming request parameters.
   */
  public function prepareForRequest(array $params = []): void {
    if (isset($params['capabilities']) && !empty($params['capabilities'])) {
      $this->setClientCapabilities($params['capabilities']);
    }

    if (isset($params['protocolVersion'])) {
      $this->setProtocolVersion($params['protocolVersion']);
    }

    if (isset($params['server_url'])) {
      $this->setMcpServerUrl($params['server_url']);
    }

    $method = $params['method'] ?? '';
    if ($method !== 'initialize' && !$this->isSessionInitialized()) {
      $this->initializeSession();
    }
  }

  /**
   * Generates a cryptographically secure request ID.
   *
   * @return string
   *   A unique, secure request identifier.
   */
  protected function generateSecureRequestId(): string {
    // Generate 8 random bytes and convert to hexadecimal.
    // This produces a 16-character string that is cryptographically secure
    // and highly unlikely to collide.
    return bin2hex(random_bytes(8));
  }

}
