<?php

namespace Drupal\api_plugins_mcp\Plugin\ApiPlugin;

use Drupal\api_plugins_mcp\McpApiPluginBase;
use Drupal\api_plugins\ApiPluginInterface;
use Drupal\api_plugins\ApiAuthenticationService;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Utility\Token;
use GuzzleHttp\Exception\GuzzleException;
use Drupal\Component\Serialization\Json;
use GuzzleHttp\ClientInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides Apify MCP Server API endpoint.
 *
 * Connects to Apify's MCP (Model Context Protocol) server to access
 * actors, runs, storage, web browser, and crawler tools.
 *
 * @ApiPlugin(
 *   id = "apify_mcp_server",
 *   label = @Translation("Apify MCP Server", context = "api_plugins"),
 *   description = @Translation("Apify MCP server for actors, runs, storage, web browser, and crawler tools.", context = "api_plugins"),
 *   type = "mcp",
 *   endpointUrl = "https://mcp.apify.com/",
 *   defaultConfig = {
 *     "vendor" = "Apify",
 *     "protocol_version" = "2025-03-26",
 *     "default_tools" = {
 *       "actors",
 *       "docs",
 *       "runs",
 *       "storage",
 *       "apify/rag-web-browser",
 *       "compass/crawler-google-places"
 *     },
 *     "client_info" = {
 *       "name" = "Drupal API Plugins",
 *       "version" = "1.0.0"
 *     }
 *   }
 * )
 */
class ApifyMcpServer extends McpApiPluginBase implements ApiPluginInterface {

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * The Apify API token.
   *
   * @var string
   */
  protected $apiToken;

  /**
   * The MCP session ID.
   *
   * @var string|null
   */
  protected $mcpSessionId;

  /**
   * Available Apify tools.
   *
   * @var array<string>
   */
  protected $apifyTools = [
    'actors',
    'docs',
    'runs',
    'storage',
    'apify/rag-web-browser',
  ];

  /**
   * {@inheritdoc}
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    LanguageManagerInterface $language_manager,
    ConfigFactoryInterface $config_factory,
    Token $token_service,
    ApiAuthenticationService $authentication_service,
    ClientInterface $http_client,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $language_manager, $config_factory, $token_service, $authentication_service);

    $this->httpClient = $http_client;

    $this->protocolVersion = '2025-03-26';
    $this->mcpServerUrl = 'https://mcp.apify.com/';
    $this->clientCapabilities = [];

    $auth_string = $this->getProviderAuthentication();
    $this->apiToken = str_replace('Bearer ', '', $auth_string);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new self(
     $configuration,
     $plugin_id,
     $plugin_definition,
     $container->get('language_manager'),
     $container->get('config.factory'),
     $container->get('token'),
     $container->get('api_plugins.authentication'),
     $container->get('http_client')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function preparePayload(array $params = []): array {
    $method = $params['method'] ?? 'initialize';

    $payload = [
      'jsonrpc' => '2.0',
      'id' => $params['id'] ?? time(),
      'method' => $method,
    ];

    switch ($method) {
      case 'initialize':
        $capabilities = $params['capabilities'] ?? $this->clientCapabilities;
        if (empty($capabilities)) {
          $capabilities = new \stdClass();
        }

        $payload['params'] = [
          'protocolVersion' => $this->protocolVersion,
          'capabilities' => $capabilities,
          'clientInfo' => $params['clientInfo'] ?? [
            'name' => 'Drupal API Plugins',
            'version' => '1.0.0',
          ],
        ];
        break;

      case 'tools/list':
        $payload['params'] = $params['params'] ?? new \stdClass();
        break;

      case 'prompts/list':
        $payload['params'] = $params['params'] ?? new \stdClass();
        break;

      case 'tools/call':
        if (empty($params['tool_name'])) {
          throw new \InvalidArgumentException('Tool name is required for tools/call method.');
        }
        $payload['params'] = [
          'name' => $params['tool_name'],
          'arguments' => $params['arguments'] ?? new \stdClass(),
        ];
        break;

      default:
        $payload['params'] = $params['params'] ?? new \stdClass();
    }

    return $payload;
  }

  /**
   * {@inheritdoc}
   */
  public function getHeaders(): array {
    if (empty($this->apiToken)) {
      throw new \Exception("Apify API token is not configured");
    }

    $headers = [
      'Authorization' => 'Bearer ' . $this->apiToken,
      'Accept' => 'application/json, text/event-stream',
      'Content-Type' => 'application/json',
      'User-Agent' => 'Drupal-API-Plugins/1.0.0',
    ];

    if (!empty($this->mcpSessionId)) {
      $headers['mcp-session-id'] = $this->mcpSessionId;
    }

    return $headers;
  }

  /**
   * {@inheritdoc}
   */
  public function getEndpoint(): string {
    if ($this->sessionInitialized) {
      return $this->mcpServerUrl;
    }

    $tools_param = implode('%2C', $this->apifyTools);
    return $this->mcpServerUrl . '?tools=' . $tools_param;
  }

  /**
   * Set API token for Apify authentication.
   *
   * @param string $token
   *   The Apify API token.
   *
   * @return $this
   *   Returns self for method chaining.
   */
  public function setApiToken(string $token): static {
    $this->apiToken = $token;
    return $this;
  }

  /**
   * Get the current API token.
   *
   * @return string
   *   The API token.
   */
  public function getApiToken(): string {
    return $this->apiToken;
  }

  /**
   * Set the tools to request from Apify MCP server.
   *
   * @param array $tools
   *   Array of tool names.
   *
   * @return $this
   *   Returns self for method chaining.
   */
  public function setApifyTools(array $tools): static {
    $this->apifyTools = $tools;
    return $this;
  }

  /**
   * Get the configured Apify tools.
   *
   * @return array
   *   Array of tool names.
   */
  public function getApifyTools(): array {
    return $this->apifyTools;
  }

  /**
   * Set MCP session ID.
   *
   * @param string|null $sessionId
   *   The MCP session ID.
   *
   * @return $this
   *   Returns self for method chaining.
   */
  public function setMcpSessionId(?string $sessionId): static {
    $this->mcpSessionId = $sessionId;
    return $this;
  }

  /**
   * Get MCP session ID.
   *
   * @return string|null
   *   The MCP session ID.
   */
  public function getMcpSessionId(): ?string {
    return $this->mcpSessionId;
  }

  /**
   * {@inheritdoc}
   */
  public function formatResponse(array $result): string|array {
    // For MCP responses, we typically want to return the structured data.
    if (isset($result['result'])) {
      return $result['result'];
    }

    if (isset($result['error'])) {
      throw new \Exception('Apify MCP Server Error: ' . ($result['error']['message'] ?? 'Unknown error'));
    }

    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public function validateResponse(array $response): bool {
    // First check HTTP status codes (throws exception for 4xx/5xx).
    parent::validateResponse($response);

    // Then validate JSON-RPC response structure.
    $data = $response['data'] ?? $response;

    // Basic JSON-RPC response validation.
    if (!isset($data['jsonrpc']) || $data['jsonrpc'] !== '2.0') {
      return FALSE;
    }

    if (!isset($data['id'])) {
      return FALSE;
    }

    // Must have either result or error.
    return isset($data['result']) || isset($data['error']);
  }

  /**
   * {@inheritdoc}
   */
  public function getAuthentication(string $plugin_id): string {
    return $this->getProviderAuthentication();
  }

  /**
   * {@inheritdoc}
   */
  protected function sendMcpRequest(array $payload): array {
    $options = [
      'headers' => $this->getHeaders(),
      'json' => $payload,
      'http_errors' => FALSE,
    // Extended timeout for MCP requests.
      'timeout' => 90,
    // Connection timeout.
      'connect_timeout' => 15,
    // Disable streaming to wait for full response.
      'stream' => FALSE,
    ];

    $url = $this->getEndpoint();

    try {
      $response = $this->httpClient->request('POST', $url, $options);
      $result = $response->getBody()->getContents();

      if ($response->hasHeader('mcp-session-id')) {
        $sessionId = $response->getHeaderLine('mcp-session-id');
        $this->setMcpSessionId($sessionId);
      }

      $decoded_data = $this->parseServerSentEvents($result);

      if (isset($decoded_data['error'])) {
        throw new \Exception('Apify MCP error: ' . ($decoded_data['error']['message'] ?? 'Unknown error'));
      }

      if (!$this->validateResponse($decoded_data)) {
        throw new \Exception('Invalid JSON-RPC response from Apify MCP server');
      }

      return $decoded_data;
    }
    catch (GuzzleException $exception) {
      throw new \Exception('Apify MCP request failed: ' . $exception->getMessage());
    }
  }

  /**
   * {@inheritdoc}
   */
  public function initializeSession(): array {
    $payload = [
      'jsonrpc' => '2.0',
      'id' => 1,
      'method' => 'initialize',
      'params' => [
        'protocolVersion' => $this->protocolVersion,
        'capabilities' => new \stdClass(),
        'clientInfo' => [
          'name' => 'Drupal API Plugins',
          'version' => '1.0.0',
        ],
      ],
    ];

    try {
      $response = $this->sendMcpRequest($payload);

      if (isset($response['result']['capabilities'])) {
        $this->serverCapabilities = $response['result']['capabilities'];
        $this->sessionInitialized = TRUE;
      }

      return $response;
    }
    catch (\Exception $e) {
      throw new \Exception('MCP session initialization failed: ' . $e->getMessage());
    }
  }

  /**
   * {@inheritdoc}
   */
  public function listTools(): array {
    if (!$this->sessionInitialized) {
      $this->initializeSession();
    }

    $payload = [
      'jsonrpc' => '2.0',
      'id' => 2,
      'method' => 'tools/list',
      'params' => new \stdClass(),
    ];

    try {
      $response = $this->sendMcpRequest($payload);

      if (isset($response['result']['tools'])) {
        $this->availableTools = $response['result']['tools'];
        return $this->availableTools;
      }

      return [];
    }
    catch (\Exception $e) {
      throw new \Exception('Failed to list MCP tools: ' . $e->getMessage());
    }
  }

  /**
   * Parse Server-Sent Events (SSE) response format.
   *
   * @param string $sse_content
   *   The SSE formatted content.
   *
   * @return array
   *   The parsed JSON data from the SSE stream.
   *
   * @throws \Exception
   *   If parsing fails.
   */
  protected function parseServerSentEvents(string $sse_content): array {
    $trimmed = trim($sse_content);

    if (strpos($trimmed, '{') === 0) {
      return Json::decode($trimmed);
    }

    $lines = explode("\n", $trimmed);
    $data = '';

    foreach ($lines as $line) {
      $line = trim($line);
      if (strpos($line, 'data: ') === 0) {
        $data = substr($line, 6);
        break;
      }
    }

    if (empty($data)) {
      throw new \Exception('No data field found in SSE response');
    }

    return Json::decode($data);
  }

  /**
   * Call a specific Apify actor via MCP.
   *
   * @param string $actorId
   *   The Apify actor ID.
   * @param array $input
   *   Input data for the actor.
   * @param array $options
   *   Additional options (timeout, memory, etc.).
   *
   * @return array
   *   Actor run result.
   *
   * @throws \Exception
   *   If actor call fails.
   */
  public function runActor(string $actorId, array $input = [], array $options = []): array {
    return $this->callTool('actors', [
      'actorId' => $actorId,
      'input' => $input,
      'options' => $options,
    ]);
  }

  /**
   * Use Apify's web browser tool for scraping.
   *
   * @param string $url
   *   URL to scrape.
   * @param array $options
   *   Scraping options.
   *
   * @return array
   *   Scraped content.
   *
   * @throws \Exception
   *   If web browser tool fails.
   */
  public function browseWeb(string $url, array $options = []): array {
    return $this->callTool('apify/rag-web-browser', [
      'url' => $url,
      'options' => $options,
    ]);
  }

  /**
   * Crawl Google Places using Compass crawler.
   *
   * @param string $query
   *   Search query.
   * @param array $options
   *   Crawling options (location, maxResults, etc.).
   *
   * @return array
   *   Places data.
   *
   * @throws \Exception
   *   If Google Places crawler fails.
   */
  public function crawlGooglePlaces(string $query, array $options = []): array {
    return $this->callTool('compass/crawler-google-places', [
      'query' => $query,
      'options' => $options,
    ]);
  }

  /**
   * Get data from Apify storage.
   *
   * @param string $storeId
   *   Storage ID.
   * @param array $options
   *   Storage access options.
   *
   * @return array
   *   Storage data.
   *
   * @throws \Exception
   *   If storage access fails.
   */
  public function getStorageData(string $storeId, array $options = []): array {
    return $this->callTool('storage', [
      'storeId' => $storeId,
      'options' => $options,
    ]);
  }

  /**
   * Get information about actor runs.
   *
   * @param string $runId
   *   Run ID to query.
   * @param array $options
   *   Query options.
   *
   * @return array
   *   Run information.
   *
   * @throws \Exception
   *   If run query fails.
   */
  public function getRunInfo(string $runId, array $options = []): array {
    return $this->callTool('runs', [
      'runId' => $runId,
      'options' => $options,
    ]);
  }

  /**
   * Get Apify documentation.
   *
   * @param string $topic
   *   Documentation topic.
   *
   * @return array
   *   Documentation content.
   *
   * @throws \Exception
   *   If documentation retrieval fails.
   */
  public function getDocumentation(string $topic): array {
    return $this->callTool('docs', [
      'topic' => $topic,
    ]);
  }

  /**
   * List available prompts from Apify MCP server.
   *
   * @return array
   *   Array of available prompts.
   *
   * @throws \Exception
   *   If listing prompts fails.
   */
  public function listPrompts(): array {
    if (!$this->sessionInitialized) {
      $this->initializeSession();
    }

    $payload = [
      'jsonrpc' => '2.0',
      'id' => $this->generateSecureRequestId(),
      'method' => 'prompts/list',
      'params' => [],
    ];

    try {
      $response = $this->sendMcpRequest($payload);

      if (isset($response['result']['prompts'])) {
        return $response['result']['prompts'];
      }

      return [];
    }
    catch (\Exception $e) {
      throw new \Exception('Failed to list Apify MCP prompts: ' . $e->getMessage());
    }
  }

}
