<?php

namespace Drupal\Tests\api_plugins_mcp\Unit;

use Drupal\api_plugins_mcp\McpApiPluginBase;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;
use Psr\Log\LoggerInterface;

/**
 * Tests for McpApiPluginBase.
 */
#[CoversClass(McpApiPluginBase::class)]
#[Group('api_plugins_mcp')]
class McpApiPluginBaseTest extends UnitTestCase {

  /**
   * The mock MCP plugin.
   *
   * @var \Drupal\api_plugins_mcp\McpApiPluginBase|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $plugin;

  /**
   * Mock logger.
   *
   * @var \Psr\Log\LoggerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $logger;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->logger = $this->createMock(LoggerInterface::class);

    $this->plugin = $this->getMockBuilder(McpApiPluginBase::class)
      ->disableOriginalConstructor()
      ->onlyMethods([
        'getLogger',
        't',
        'sendMcpRequest',
        'preparePayload',
        'getHeaders',
        'formatResponse',
        'getAuthentication',
      ])
      ->getMock();

    $this->plugin->method('getLogger')
      ->willReturn($this->logger);

    $this->plugin->method('t')
      ->willReturnCallback(function ($string) {
        return $string;
      });
  }

  /**
   * Tests setMcpServerUrl with valid HTTPS URL.
   */
  public function testSetMcpServerUrlWithValidHttpsUrl(): void {
    $url = 'https://mcp.apify.com/';

    $this->logger->expects($this->once())
      ->method('info')
      ->with(
        'MCP server URL validated: @url',
        ['@url' => $url]
      );

    $result = $this->plugin->setMcpServerUrl($url);

    $this->assertSame($this->plugin, $result);
    $this->assertEquals($url, $this->plugin->getMcpServerUrl());
  }

  /**
   * Tests setMcpServerUrl with valid HTTP URL.
   */
  public function testSetMcpServerUrlWithValidHttpUrl(): void {
    $url = 'http://example.com/mcp';

    $this->logger->expects($this->once())
      ->method('info');

    $result = $this->plugin->setMcpServerUrl($url);

    $this->assertSame($this->plugin, $result);
  }

  /**
   * Tests setMcpServerUrl throws exception for invalid URL format.
   */
  public function testSetMcpServerUrlThrowsExceptionForInvalidFormat(): void {
    $this->expectException(\InvalidArgumentException::class);
    $this->expectExceptionMessage('Invalid URL format');

    $this->plugin->setMcpServerUrl('not-a-valid-url');
  }

  /**
   * Tests setMcpServerUrl throws exception for unsupported scheme.
   */
  public function testSetMcpServerUrlThrowsExceptionForUnsupportedScheme(): void {
    $this->expectException(\InvalidArgumentException::class);
    $this->expectExceptionMessage('Only HTTP and HTTPS schemes are allowed');

    $this->plugin->setMcpServerUrl('ftp://example.com/mcp');
  }

  /**
   * Tests setMcpServerUrl throws exception for file:// scheme.
   */
  public function testSetMcpServerUrlThrowsExceptionForFileScheme(): void {
    $this->expectException(\InvalidArgumentException::class);
    $this->expectExceptionMessage('Invalid URL format.');

    $this->plugin->setMcpServerUrl('file:///etc/passwd');
  }

  /**
   * Tests setMcpServerUrl throws exception for localhost.
   */
  public function testSetMcpServerUrlThrowsExceptionForLocalhost(): void {
    $this->expectException(\InvalidArgumentException::class);
    $this->expectExceptionMessage('Cannot access localhost.');

    $this->plugin->setMcpServerUrl('http://127.0.0.1/mcp');
  }

  /**
   * Tests setMcpServerUrl throws exception for private IP.
   */
  #[DataProvider('privateIpProvider')]
  public function testSetMcpServerUrlThrowsExceptionForPrivateIp(string $ip): void {
    $this->expectException(\InvalidArgumentException::class);
    $this->expectExceptionMessage('Cannot access private or reserved IP addresses');

    $this->plugin->setMcpServerUrl("http://{$ip}/mcp");
  }

  /**
   * Data provider for private IP tests.
   */
  public static function privateIpProvider(): array {
    return [
      'private class A' => ['10.0.0.1'],
      'private class B' => ['172.16.0.1'],
      'private class C' => ['192.168.1.1'],
      'link-local' => ['169.254.1.1'],
    ];
  }

  /**
   * Tests getMcpServerUrl returns set URL.
   */
  public function testGetMcpServerUrl(): void {
    $url = 'https://8.8.8.8/';

    $this->plugin->setMcpServerUrl($url);

    $this->assertEquals($url, $this->plugin->getMcpServerUrl());
  }

  /**
   * Tests setProtocolVersion and getProtocolVersion.
   */
  public function testProtocolVersionGetterSetter(): void {
    $version = '2025-03-26';

    $result = $this->plugin->setProtocolVersion($version);

    $this->assertSame($this->plugin, $result);
    $this->assertEquals($version, $this->plugin->getProtocolVersion());
  }

  /**
   * Tests setClientCapabilities and getClientCapabilities.
   */
  public function testClientCapabilitiesGetterSetter(): void {
    $capabilities = [
      'sampling' => TRUE,
      'tools' => TRUE,
    ];

    $result = $this->plugin->setClientCapabilities($capabilities);

    $this->assertSame($this->plugin, $result);
    $this->assertEquals($capabilities, $this->plugin->getClientCapabilities());
  }

  /**
   * Tests getServerCapabilities returns empty array by default.
   */
  public function testGetServerCapabilitiesDefault(): void {
    $this->assertEquals([], $this->plugin->getServerCapabilities());
  }

  /**
   * Tests isSessionInitialized returns FALSE by default.
   */
  public function testIsSessionInitializedDefaultFalse(): void {
    $this->assertFalse($this->plugin->isSessionInitialized());
  }

  /**
   * Tests URL validation logs successful validation.
   */
  public function testValidateUrlLogsSuccess(): void {
    $url = 'https://8.8.8.8/mcp';

    $this->logger->expects($this->once())
      ->method('info')
      ->with(
        'MCP server URL validated: @url',
        ['@url' => $url]
      );

    $this->plugin->setMcpServerUrl($url);
  }

  /**
   * Tests setMcpServerUrl with URL containing port.
   */
  public function testSetMcpServerUrlWithPort(): void {
    $url = 'https://8.8.8.8:8080/mcp';

    $result = $this->plugin->setMcpServerUrl($url);

    $this->assertSame($this->plugin, $result);
    $this->assertEquals($url, $this->plugin->getMcpServerUrl());
  }

  /**
   * Tests setMcpServerUrl with URL containing query parameters.
   */
  public function testSetMcpServerUrlWithQueryParams(): void {
    $url = 'https://mcp.apify.com/?tools=actors,docs';

    $result = $this->plugin->setMcpServerUrl($url);

    $this->assertSame($this->plugin, $result);
  }

  /**
   * Tests setMcpServerUrl throws exception for URL without host.
   */
  public function testSetMcpServerUrlThrowsExceptionForMissingHost(): void {
    $this->expectException(\InvalidArgumentException::class);
    $this->expectExceptionMessage('Invalid URL format');

    $this->plugin->setMcpServerUrl('https:///path');
  }

  /**
   * Tests setMcpServerUrl throws exception for URL without scheme.
   */
  public function testSetMcpServerUrlThrowsExceptionForMissingScheme(): void {
    $this->expectException(\InvalidArgumentException::class);
    $this->expectExceptionMessage('Invalid URL format');

    $this->plugin->setMcpServerUrl('//example.com/mcp');
  }

}
