# API Plugins - OpenAI

OpenAI API integration for API Plugins module. Provides access to OpenAI's Chat Completions and Embeddings endpoints.

## Features

- **Chat Completions** - GPT-4, GPT-4 Turbo, GPT-3.5 Turbo models
- **Embeddings** - Text vectorization for semantic search and RAG applications
- Full message history support
- System prompts and roles (system, user, assistant)
- Configurable parameters (temperature, max_tokens, top_p, etc.)
- Token replacement support
- Response streaming (coming soon)

## Installation

```bash
drush en api_plugins_openai
```

**Requires:**
- api_plugins (core module)
- Valid OpenAI API key

## Configuration

### Setting Up Your API Key

You have two options for configuring your OpenAI API key:

#### Option 1: Using Environment Variables (Recommended)

Set the `OPENAI_API_KEY` environment variable:

```bash
# In your .env file or shell
export OPENAI_API_KEY="sk-..."
```

#### Option 2: Using Drupal Key Module (Most Secure)

1. Install and enable the Key module:
   ```bash
   composer require drupal/key
   drush en key
   ```

2. Navigate to Configuration > System > Keys (`/admin/config/system/keys`)

3. Add a new key with your OpenAI API key

4. Configure API Plugins to use this key at `/admin/config/api-plugins/settings`

## Usage

### Basic Chat Completion

```php
use Drupal\api_plugins\ApiRequestService;

/** @var \Drupal\api_plugins\ApiRequestService $api_request */
$api_request = \Drupal::service('api_plugins.request');
$response = $api_request->sendRequest('openai_chat_completions', [
  'system_prompt' => 'You are a helpful assitamt',
  'messages' => [
    ['role' => 'user', 'content' => 'What is main city of Czechia'],
  ],
]);

echo $response; // Returns the assistant's response text
```

### Chat with System Prompt

```php
$response = $api_request->sendRequest('openai_chat_completions', [
  'system_prompt' => 'You are a helpful Drupal expert.',
  'messages' => [
    ['role' => 'user', 'content' => 'How do I create a custom block?'],
  ],
]);

echo $response;
```

### Multi-Turn Conversation

```php
$response = $api_request->sendRequest('openai_chat_completions', [
  'messages' => [
    ['role' => 'user', 'content' => 'What is dependency injection?'],
    ['role' => 'assistant', 'content' => 'Dependency injection is a design pattern...'],
    ['role' => 'user', 'content' => 'Can you show me an example in Drupal?'],
  ],
]);

echo $response;
```

### Adjusting Model Parameters

```php
$response = $api_request->sendRequest('openai_chat_completions', [
  'model' => 'gpt-4o',
  'temperature' => 0.7,
  'max_tokens' => 2000,
  'top_p' => 0.9,
  'messages' => [
    ['role' => 'user', 'content' => 'Write a creative story about a robot.'],
  ],
]);

echo $response;
```

### Using Different Models

```php
// GPT-4o (Latest, most capable)
$response = $api_request->sendRequest('openai_chat_completions', [
  'model' => 'gpt-4o',
  'messages' => [
    ['role' => 'user', 'content' => 'Analyze this complex code...'],
  ],
]);

// GPT-4o-mini (Fast and affordable)
$response = $api_request->sendRequest('openai_chat_completions', [
  'model' => 'gpt-4o-mini',
  'messages' => [
    ['role' => 'user', 'content' => 'Summarize this article...'],
  ],
]);

// GPT-3.5 Turbo (Legacy, most affordable)
$response = $api_request->sendRequest('openai_chat_completions', [
  'model' => 'gpt-3.5-turbo',
  'messages' => [
    ['role' => 'user', 'content' => 'Quick question...'],
  ],
]);
```

## Embeddings API

Use embeddings to convert text into vector representations for semantic search, recommendations, and RAG applications.

### Generate Embeddings

```php
use Drupal\api_plugins\ApiRequestService;

/** @var \Drupal\api_plugins\ApiRequestService $api_request */
$api_request = \Drupal::service('api_plugins.request');

$result = $api_request->sendRequest('openai_embeddings', [
  'input' => 'Drupal is a powerful content management system.',
]);

// Returns array with:
// - embeddings: array of 1536 float values (for text-embedding-3-small)
// - model: the model used
// - usage: token usage information
print_r($result);
```

### Embedding Models

```php
// Small model (1536 dimensions, most affordable)
$result = $api_request->sendRequest('openai_embeddings', [
  'model' => 'text-embedding-3-small',
  'input' => 'Your text here',
]);

// Large model (3072 dimensions, highest quality)
$result = $api_request->sendRequest('openai_embeddings', [
  'model' => 'text-embedding-3-large',
  'input' => 'Your text here',
]);
```

### Sanitize HTML Before Embedding

```php
$result = $api_request->sendRequest('openai_embeddings', [
  'input' => '<p>This is <strong>HTML</strong> content</p>',
  'sanitize' => TRUE, // Strips HTML tags
]);
```

## Using the Plugin Manager Directly

For more control, you can use the plugin manager directly:

### Chat Completions

```php
$plugin_manager = \Drupal::service('plugin.manager.api_endpoint');

/** @var \Drupal\api_plugins_openai\Plugin\ApiPlugin\OpenAiChatCompletions $plugin */
$plugin = $plugin_manager->createInstance('openai_chat_completions');

// Configure the plugin
$plugin
  ->setModel('gpt-4o')
  ->setTemperature(0.7)
  ->setMaxTokens(2000)
  ->setSystemPrompt('You are a helpful coding assistant.');

// Prepare payload
$payload = $plugin->preparePayload([
  'messages' => [
    ['role' => 'user', 'content' => 'Explain services in Drupal.'],
  ],
]);

// Send request using ApiRequestService
$api_request = \Drupal::service('api_plugins.request');
$response = $api_request->sendRequest('openai_chat_completions', [
  'messages' => [
    ['role' => 'user', 'content' => 'Explain services in Drupal.'],
  ],
]);

echo $response;
```

### Embeddings

```php
$plugin_manager = \Drupal::service('plugin.manager.api_endpoint');

/** @var \Drupal\api_plugins_openai\Plugin\ApiPlugin\OpenAiEmbeddings $plugin */
$plugin = $plugin_manager->createInstance('openai_embeddings');

// Configure model
$plugin->setModel('text-embedding-3-large');

// Generate embeddings
$result = $api_request->sendRequest('openai_embeddings', [
  'input' => 'Content to embed',
]);

// Access the embedding vector
$vector = $result['embeddings'];
// Use $vector for similarity search, clustering, etc.
```

## Available Models

### Chat Models

| Model | Description | Context Window | Best For |
|-------|-------------|----------------|----------|
| `gpt-4o` | Latest GPT-4 Omni model | 128K tokens | Complex tasks, multimodal |
| `gpt-4o-mini` | Smaller, faster GPT-4 | 128K tokens | Most use cases |
| `gpt-4-turbo` | GPT-4 Turbo | 128K tokens | High-quality responses |
| `gpt-3.5-turbo` | Legacy GPT-3.5 | 16K tokens | Simple tasks |

### Embedding Models

| Model | Dimensions | Cost | Best For |
|-------|------------|------|----------|
| `text-embedding-3-small` | 1536 | Low | Most use cases |
| `text-embedding-3-large` | 3072 | Medium | Highest quality |
| `text-embedding-ada-002` | 1536 | Low | Legacy applications |

## Configuration Parameters

### Chat Completions

| Parameter | Type | Default | Description |
|-----------|------|---------|-------------|
| `model` | string | `gpt-4o-mini` | The model to use |
| `temperature` | float | `0.0` | Randomness (0.0-2.0) |
| `max_tokens` | int | `null` | Maximum response tokens |
| `top_p` | float | `1.0` | Nucleus sampling (0.0-1.0) |
| `frequency_penalty` | float | `0.0` | Reduce repetition (-2.0 to 2.0) |
| `presence_penalty` | float | `0.0` | Encourage new topics (-2.0 to 2.0) |
| `system_prompt` | string | `''` | System instructions |
| `messages` | array | required | Conversation history |

### Embeddings

| Parameter | Type | Default | Description |
|-----------|------|---------|-------------|
| `model` | string | `text-embedding-3-small` | Embedding model |
| `input` | string | required | Text to embed (max 32,000 chars) |
| `sanitize` | bool | `false` | Strip HTML tags |
| `encoding_format` | string | `float` | Output format |

## Token Management

### Understanding Tokens

- Tokens are pieces of words (e.g., "Drupal" = 1 token, "tokenization" = 3 tokens)
- Average: 1 token ≈ 4 characters or ≈ 0.75 words
- Both input and output count toward your usage

### Estimating Costs

```php
// Example: Approximately 100 tokens
$short_message = [
  'messages' => [
    ['role' => 'user', 'content' => 'What is Drupal?'], // ~5 tokens
  ],
  'max_tokens' => 100, // Limit response to ~100 tokens
];

// Use gpt-4o-mini for cost-effective responses
$response = $api_request->sendRequest('openai_chat_completions', array_merge(
  $short_message,
  ['model' => 'gpt-4o-mini']
));
```

## Error Handling

```php
use Drupal\api_plugins\Exception\ApiAuthenticationException;
use Drupal\api_plugins\Exception\ApiRateLimitException;
use Drupal\api_plugins\Exception\ApiResponseException;

try {
  $response = $api_request->sendRequest('openai_chat_completions', [
    'messages' => [
      ['role' => 'user', 'content' => 'Hello!'],
    ],
  ]);

  echo $response;
}
catch (ApiAuthenticationException $e) {
  // API key is invalid or missing
  \Drupal::logger('my_module')->error('OpenAI authentication failed: @error', [
    '@error' => $e->getMessage(),
  ]);
}
catch (ApiRateLimitException $e) {
  // Rate limit exceeded
  $retry_after = $e->getRetryAfter();
  \Drupal::logger('my_module')->warning('Rate limit exceeded. Retry after @seconds seconds.', [
    '@seconds' => $retry_after,
  ]);
}
catch (ApiResponseException $e) {
  // Other API errors (400, 500, etc.)
  \Drupal::logger('my_module')->error('OpenAI API error: @error', [
    '@error' => $e->getMessage(),
  ]);
}
```

## Advanced Examples

### Semantic Search with Embeddings

```php
// 1. Generate embeddings for your content
$documents = [
  'Drupal is a content management system.',
  'WordPress is a blogging platform.',
  'Joomla is a CMS framework.',
];

$embeddings = [];
foreach ($documents as $doc) {
  $result = $api_request->sendRequest('openai_embeddings', [
    'input' => $doc,
  ]);
  $embeddings[] = $result['embeddings'];
}

// 2. Generate embedding for search query
$query = 'content management';
$query_result = $api_request->sendRequest('openai_embeddings', [
  'input' => $query,
]);
$query_embedding = $query_result['embeddings'];

// 3. Calculate cosine similarity and find most relevant document
function cosine_similarity($a, $b) {
  $dot_product = array_sum(array_map(fn($x, $y) => $x * $y, $a, $b));
  $magnitude_a = sqrt(array_sum(array_map(fn($x) => $x * $x, $a)));
  $magnitude_b = sqrt(array_sum(array_map(fn($x) => $x * $x, $b)));
  return $dot_product / ($magnitude_a * $magnitude_b);
}

$similarities = [];
foreach ($embeddings as $i => $embedding) {
  $similarities[$i] = cosine_similarity($query_embedding, $embedding);
}

arsort($similarities);
echo "Most relevant: " . $documents[key($similarities)];
```

### RAG (Retrieval Augmented Generation)

```php
// 1. Find relevant context using embeddings (from above)
$relevant_context = $documents[key($similarities)];

// 2. Use context in chat completion
$response = $api_request->sendRequest('openai_chat_completions', [
  'system_prompt' => 'Answer questions based on the following context: ' . $relevant_context,
  'messages' => [
    ['role' => 'user', 'content' => 'What is the main topic?'],
  ],
]);

echo $response;
```

### Content Moderation

```php
$user_input = $_POST['comment'];

// Check if content is appropriate
$response = $api_request->sendRequest('openai_chat_completions', [
  'model' => 'gpt-4o-mini',
  'system_prompt' => 'You are a content moderator. Reply with "SAFE" or "UNSAFE" and explain why.',
  'messages' => [
    ['role' => 'user', 'content' => $user_input],
  ],
]);

if (strpos($response, 'UNSAFE') !== FALSE) {
  drupal_set_message('Your comment contains inappropriate content.', 'error');
}
```

### Token Replacement in System Prompts

```php
$response = $api_request->sendRequest('openai_chat_completions', [
  'system_prompt' => 'You are helping [user:name] with Drupal development.',
  'token_data' => [
    'user' => \Drupal::currentUser(),
  ],
  'messages' => [
    ['role' => 'user', 'content' => 'How do I create a module?'],
  ],
]);
```

## Security Features

This module implements several security measures:

### API Key Protection
- Keys stored securely via Key module
- Sanitized to prevent header injection
- Validated for proper format and length

### Input Validation
- Model names validated against allowed patterns
- Input text length limits enforced (32,000 chars for embeddings)
- HTML sanitization available for embeddings

### Prompt Injection Prevention
- System prompts filtered for common injection patterns
- Token replacement with sanitization enabled by default
- Suspicious patterns logged for review

### Error Handling
- Detailed errors logged for administrators
- Safe, generic messages shown to users
- Specific exception types for different error scenarios

## Performance Tips

1. **Use gpt-4o-mini** for most tasks - it's fast and affordable
2. **Set max_tokens** to limit response length and costs
3. **Cache embeddings** - they don't change for the same text
4. **Use temperature=0** for consistent, deterministic responses
5. **Batch operations** when possible to reduce API calls

## Troubleshooting

### "API authentication failed"
- Check that your API key is set correctly
- Verify the key hasn't expired or been revoked
- Ensure the key has the correct permissions

### "Rate limit exceeded"
- Implement request throttling in your application
- Consider upgrading your OpenAI plan
- Use caching to reduce API calls

### "Maximum context length exceeded"
- Reduce the length of your messages
- Summarize long conversations
- Use a model with larger context window (GPT-4)

### "Invalid model name"
- Check spelling of model name
- Ensure you're using a supported model
- Some models require specific API tier access

### Empty or incomplete responses
- Increase `max_tokens` parameter
- Check if content filters triggered
- Review your system prompt for conflicts

## Resources

- [OpenAI API Documentation](https://platform.openai.com/docs/api-reference)
- [OpenAI Models Overview](https://platform.openai.com/docs/models)
- [Best Practices](https://platform.openai.com/docs/guides/production-best-practices)
- [API Plugins Documentation](../../../README.md)

## Support

For issues related to:
- **This module**: Create an issue in the project repository
- **OpenAI API**: Check [OpenAI's status page](https://status.openai.com/)
- **API keys**: Visit [OpenAI Platform](https://platform.openai.com/account/api-keys)
