<?php

namespace Drupal\api_plugins_openai\Plugin\ApiPlugin;

use Drupal\api_plugins\AiApiPluginBase;
use Drupal\api_plugins\ApiPluginInterface;

/**
 * Provides OpenAI Chat Completions API endpoint.
 *
 * @ApiPlugin(
 *   id = "openai_chat_completions",
 *   label = @Translation("OpenAI Chat", context = "api_plugins"),
 *   description = @Translation("OpenAI Chat Completions endpoint.", context = "api_plugins"),
 *   type = "ai",
 *   endpointUrl = "https://api.openai.com/v1/chat/completions",
 *   defaultConfig = {
 *     "vendor" = "OpenAI",
 *     "model" = "gpt-4o-mini",
 *     "temperature" = 0.0
 *   }
 * )
 */
class OpenAiChatCompletions extends AiApiPluginBase implements ApiPluginInterface {

  /**
   * {@inheritdoc}
   */
  protected string $model = 'gpt-4o-mini';

  /**
   * {@inheritdoc}
   */
  protected float $temperature = 0.0;

  /**
   * {@inheritdoc}
   */
  public function prepareForRequest(array $params = []): void {
    $this->setEndpoint('https://api.openai.com/v1/chat/completions');
    parent::prepareForRequest($params);
  }

  /**
   * {@inheritdoc}
   */
  public function preparePayload(array $params = []): array {
    $messages = $this->createMessages($params);

    if (empty($messages)) {
      throw new \InvalidArgumentException('At least one message is required for chat completions.');
    }

    $payload = $this->getAiParameters();
    $payload['messages'] = $messages;

    return $payload;
  }

  /**
   * {@inheritdoc}
   */
  public function listModels(): array {
    return [
      'gpt-4o',
      'gpt-4o-mini',
      'gpt-4-turbo',
      'gpt-3.5-turbo',
    ];
  }

  /**
   * Format the API response.
   *
   * @param array $result
   *   The API response.
   *
   * @return string
   *   Formatted response text.
   */
  public function formatResponse(array $result): string {
    if (isset($result['choices'][0]['message']['content'])) {
      return $result['choices'][0]['message']['content'];
    }

    return '';
  }

}
