<?php

namespace Drupal\api_plugins_openai\Plugin\ApiPlugin;

use Drupal\api_plugins\AiApiPluginBase;
use Drupal\api_plugins\ApiPluginInterface;

/**
 * Provides OpenAI Embeddings API endpoint.
 *
 * @ApiPlugin(
 *   id = "openai_embeddings",
 *   label = @Translation("OpenAI Embeddings", context = "api_plugins"),
 *   description = @Translation("OpenAI Embeddings endpoint for text vectorization.", context = "api_plugins"),
 *   type = "ai",
 *   endpointUrl = "https://api.openai.com/v1/embeddings",
 *   defaultConfig = {
 *     "vendor" = "OpenAI",
 *     "model" = "text-embedding-3-small"
 *   }
 * )
 */
class OpenAiEmbeddings extends AiApiPluginBase implements ApiPluginInterface {

  /**
   * {@inheritdoc}
   */
  protected string $model = 'text-embedding-3-small';

  /**
   * {@inheritdoc}
   *
   * @throws \InvalidArgumentException
   *   If input text is missing or invalid.
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   *   If input validation fails.
   */
  public function preparePayload(array $params = []): array {
    $input = $params['input'] ?? '';

    if (!is_string($input)) {
      throw new \InvalidArgumentException('Input must be a string.');
    }

    if (empty($input)) {
      throw new \InvalidArgumentException('Input text is required for embeddings.');
    }

    $max_length = 32000;
    if (strlen($input) > $max_length) {
      throw new \InvalidArgumentException("Input exceeds maximum length of $max_length characters.");
    }

    if (!empty($params['sanitize'])) {
      $input = strip_tags($input);
    }

    $payload = $this->getAiParameters();
    $payload['input'] = $input;
    $payload['encoding_format'] = 'float';

    return $payload;
  }

  /**
   * Format the embeddings API response.
   *
   * @param array $result
   *   The API response.
   *
   * @return array
   *   Formatted embeddings data.
   */
  public function formatResponse(array $result): array {
    if (isset($result['data'][0]['embedding'])) {
      return [
        'embeddings' => $result['data'][0]['embedding'],
        'model' => $result['model'] ?? $this->model,
        'usage' => $result['usage'] ?? [],
      ];
    }

    return ['embeddings' => [], 'error' => 'Invalid embeddings response'];
  }

  /**
   * {@inheritdoc}
   */
  public function listModels(): array {
    return [
      'text-embedding-3-small',
      'text-embedding-3-large',
      'text-embedding-ada-002',
    ];
  }

  /**
   * Validate embeddings response.
   *
   * @param array $response
   *   The response to validate.
   *
   * @return bool
   *   TRUE if valid embeddings response.
   */
  public function validateResponse(array $response): bool {
    parent::validateResponse($response);

    $data = $response['data'] ?? $response;
    return isset($data['embeddings']) && is_array($data['embeddings']) && !empty($data['embeddings']);
  }

}
