<?php

namespace Drupal\Tests\api_plugins_openai\Unit\Plugin\ApiPlugin;

use Drupal\api_plugins_openai\Plugin\ApiPlugin\OpenAiChatCompletions;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests for OpenAiChatCompletions plugin.
 */
#[CoversClass(OpenAiChatCompletions::class)]
#[Group('api_plugins_openai')]
class OpenAiChatCompletionsTest extends UnitTestCase {

  /**
   * The OpenAI Chat Completions plugin.
   *
   * @var \Drupal\api_plugins_openai\Plugin\ApiPlugin\OpenAiChatCompletions|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $plugin;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->plugin = $this->getMockBuilder(OpenAiChatCompletions::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['createMessages', 't'])
      ->getMock();

    // Mock translation.
    $this->plugin->method('t')
      ->willReturnCallback(function ($string) {
        return $string;
      });
  }

  /**
   * Tests preparePayload with valid messages.
   */
  public function testPreparePayloadWithValidMessages(): void {
    $messages = [
      ['role' => 'system', 'content' => 'You are helpful'],
      ['role' => 'user', 'content' => 'Hello'],
    ];

    $this->plugin->method('createMessages')
      ->willReturn($messages);

    $payload = $this->plugin->preparePayload(['prompt' => 'Hello']);

    $this->assertIsArray($payload);
    $this->assertArrayHasKey('messages', $payload);
    $this->assertArrayHasKey('model', $payload);
    $this->assertEquals($messages, $payload['messages']);
  }

  /**
   * Tests preparePayload throws exception for empty messages.
   */
  public function testPreparePayloadThrowsExceptionForEmptyMessages(): void {
    $this->plugin->method('createMessages')
      ->willReturn([]);

    $this->expectException(\InvalidArgumentException::class);
    $this->expectExceptionMessage('At least one message is required');

    $this->plugin->preparePayload([]);
  }

  /**
   * Tests preparePayload includes model parameter.
   */
  public function testPreparePayloadIncludesModel(): void {
    $messages = [['role' => 'user', 'content' => 'Test']];

    $this->plugin->method('createMessages')
      ->willReturn($messages);

    $payload = $this->plugin->preparePayload(['prompt' => 'Test']);

    $this->assertArrayHasKey('model', $payload);
  }

  /**
   * Tests preparePayload includes temperature.
   */
  public function testPreparePayloadIncludesTemperature(): void {
    $messages = [['role' => 'user', 'content' => 'Test']];

    $this->plugin->method('createMessages')
      ->willReturn($messages);

    $payload = $this->plugin->preparePayload(['prompt' => 'Test']);

    $this->assertArrayHasKey('temperature', $payload);
  }

  /**
   * Tests listModels returns available chat models.
   */
  public function testListModels(): void {
    $models = $this->plugin->listModels();

    $this->assertIsArray($models);
    $this->assertNotEmpty($models);
    $this->assertContains('gpt-4o', $models);
    $this->assertContains('gpt-4o-mini', $models);
    $this->assertContains('gpt-4-turbo', $models);
    $this->assertContains('gpt-3.5-turbo', $models);
  }

  /**
   * Tests formatResponse extracts message content.
   */
  public function testFormatResponseExtractsContent(): void {
    $result = [
      'choices' => [
        [
          'message' => [
            'role' => 'assistant',
            'content' => 'Hello! How can I help you?',
          ],
          'finish_reason' => 'stop',
        ],
      ],
      'model' => 'gpt-4o',
      'usage' => [
        'prompt_tokens' => 10,
        'completion_tokens' => 8,
        'total_tokens' => 18,
      ],
    ];

    $formatted = $this->plugin->formatResponse($result);

    $this->assertEquals('Hello! How can I help you?', $formatted);
  }

  /**
   * Tests formatResponse returns empty string for missing content.
   */
  public function testFormatResponseReturnsEmptyForMissingContent(): void {
    $result = ['choices' => []];

    $formatted = $this->plugin->formatResponse($result);

    $this->assertEquals('', $formatted);
  }

  /**
   * Tests formatResponse with malformed response.
   */
  public function testFormatResponseWithMalformedResponse(): void {
    $result = ['error' => 'Something went wrong'];

    $formatted = $this->plugin->formatResponse($result);

    $this->assertEquals('', $formatted);
  }

  /**
   * Tests preparePayload with custom messages parameter.
   */
  public function testPreparePayloadWithCustomMessages(): void {
    $custom_messages = [
      ['role' => 'system', 'content' => 'Custom system'],
      ['role' => 'user', 'content' => 'Custom user'],
      ['role' => 'assistant', 'content' => 'Custom assistant'],
      ['role' => 'user', 'content' => 'Follow-up'],
    ];

    $this->plugin->method('createMessages')
      ->willReturn($custom_messages);

    $payload = $this->plugin->preparePayload([
      'messages' => $custom_messages,
    ]);

    $this->assertCount(4, $payload['messages']);
  }

  /**
   * Tests default model is gpt-4o-mini.
   */
  public function testDefaultModelIsGpt4oMini(): void {
    $messages = [['role' => 'user', 'content' => 'Test']];

    $this->plugin->method('createMessages')
      ->willReturn($messages);

    // Use reflection to access protected model property.
    $reflection = new \ReflectionClass($this->plugin);
    $property = $reflection->getProperty('model');
    $property->setAccessible(TRUE);

    $model = $property->getValue($this->plugin);

    $this->assertEquals('gpt-4o-mini', $model);
  }

  /**
   * Tests default temperature is 0.0.
   */
  public function testDefaultTemperatureIsZero(): void {
    $messages = [['role' => 'user', 'content' => 'Test']];

    $this->plugin->method('createMessages')
      ->willReturn($messages);

    // Use reflection to access protected temperature property.
    $reflection = new \ReflectionClass($this->plugin);
    $property = $reflection->getProperty('temperature');
    $property->setAccessible(TRUE);

    $temperature = $property->getValue($this->plugin);

    $this->assertEquals(0.0, $temperature);
  }

}
