<?php

namespace Drupal\Tests\api_plugins_openai\Unit\Plugin\ApiPlugin;

use Drupal\api_plugins_openai\Plugin\ApiPlugin\OpenAiEmbeddings;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests for OpenAiEmbeddings plugin.
 */
#[CoversClass(OpenAiEmbeddings::class)]
#[Group('api_plugins_openai')]
class OpenAiEmbeddingsTest extends UnitTestCase {

  /**
   * The OpenAI Embeddings plugin.
   *
   * @var \Drupal\api_plugins_openai\Plugin\ApiPlugin\OpenAiEmbeddings|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $plugin;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->plugin = $this->getMockBuilder(OpenAiEmbeddings::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['t'])
      ->getMock();

    // Mock translation.
    $this->plugin->method('t')
      ->willReturnCallback(function ($string) {
        return $string;
      });
  }

  /**
   * Tests preparePayload with valid input.
   */
  public function testPreparePayloadWithValidInput(): void {
    $params = [
      'input' => 'Test embedding text',
    ];

    $payload = $this->plugin->preparePayload($params);

    $this->assertIsArray($payload);
    $this->assertArrayHasKey('input', $payload);
    $this->assertArrayHasKey('encoding_format', $payload);
    $this->assertArrayHasKey('model', $payload);
    $this->assertEquals('Test embedding text', $payload['input']);
    $this->assertEquals('float', $payload['encoding_format']);
  }

  /**
   * Tests preparePayload throws exception for empty input.
   */
  public function testPreparePayloadThrowsExceptionForEmptyInput(): void {
    $this->expectException(\InvalidArgumentException::class);
    $this->expectExceptionMessage('Input text is required for embeddings.');

    $this->plugin->preparePayload(['input' => '']);
  }

  /**
   * Tests preparePayload throws exception for non-string input.
   */
  public function testPreparePayloadThrowsExceptionForNonStringInput(): void {
    $this->expectException(\InvalidArgumentException::class);
    $this->expectExceptionMessage('Input must be a string.');

    $this->plugin->preparePayload(['input' => 12345]);
  }

  /**
   * Tests preparePayload throws exception for input exceeding max length.
   */
  public function testPreparePayloadThrowsExceptionForTooLongInput(): void {
    $this->expectException(\InvalidArgumentException::class);
    $this->expectExceptionMessage('Input exceeds maximum length');

    // Create a string longer than 32,000 characters.
    $long_input = str_repeat('a', 32001);

    $this->plugin->preparePayload(['input' => $long_input]);
  }

  /**
   * Tests preparePayload sanitizes input when requested.
   */
  public function testPreparePayloadSanitizesInput(): void {
    $params = [
      'input' => '<script>alert("xss")</script>Test text',
      'sanitize' => TRUE,
    ];

    $payload = $this->plugin->preparePayload($params);

    $this->assertEquals('alert("xss")Test text', $payload['input']);
    $this->assertStringNotContainsString('<script>', $payload['input']);
  }

  /**
   * Tests preparePayload with missing input parameter.
   */
  public function testPreparePayloadWithMissingInput(): void {
    $this->expectException(\InvalidArgumentException::class);

    $this->plugin->preparePayload([]);
  }

  /**
   * Tests formatResponse with valid embeddings data.
   */
  public function testFormatResponseWithValidData(): void {
    $result = [
      'data' => [
        [
          'embedding' => [0.1, 0.2, 0.3, 0.4],
          'index' => 0,
        ],
      ],
      'model' => 'text-embedding-3-small',
      'usage' => [
        'prompt_tokens' => 10,
        'total_tokens' => 10,
      ],
    ];

    $formatted = $this->plugin->formatResponse($result);

    $this->assertIsArray($formatted);
    $this->assertArrayHasKey('embeddings', $formatted);
    $this->assertArrayHasKey('model', $formatted);
    $this->assertArrayHasKey('usage', $formatted);
    $this->assertEquals([0.1, 0.2, 0.3, 0.4], $formatted['embeddings']);
    $this->assertEquals('text-embedding-3-small', $formatted['model']);
  }

  /**
   * Tests formatResponse with missing embeddings.
   */
  public function testFormatResponseWithMissingEmbeddings(): void {
    $result = ['data' => []];

    $formatted = $this->plugin->formatResponse($result);

    $this->assertIsArray($formatted);
    $this->assertArrayHasKey('error', $formatted);
    $this->assertEquals('Invalid embeddings response', $formatted['error']);
  }

  /**
   * Tests listModels returns available embedding models.
   */
  public function testListModels(): void {
    $models = $this->plugin->listModels();

    $this->assertIsArray($models);
    $this->assertNotEmpty($models);
    $this->assertContains('text-embedding-3-small', $models);
    $this->assertContains('text-embedding-3-large', $models);
    $this->assertContains('text-embedding-ada-002', $models);
  }

  /**
   * Tests validateResponse with valid embeddings response.
   */
  public function testValidateResponseWithValidData(): void {
    $response = [
      'data' => [
        'embeddings' => [0.1, 0.2, 0.3],
      ],
    ];

    $result = $this->plugin->validateResponse($response);

    $this->assertTrue($result);
  }

  /**
   * Tests validateResponse with empty embeddings.
   */
  public function testValidateResponseWithEmptyEmbeddings(): void {
    $response = [
      'data' => [
        'embeddings' => [],
      ],
    ];

    $result = $this->plugin->validateResponse($response);

    $this->assertFalse($result);
  }

  /**
   * Tests validateResponse with missing embeddings.
   */
  public function testValidateResponseWithMissingEmbeddings(): void {
    $response = ['data' => []];

    $result = $this->plugin->validateResponse($response);

    $this->assertFalse($result);
  }

  /**
   * Tests preparePayload with maximum valid length input.
   */
  public function testPreparePayloadWithMaximumLength(): void {
    // Test with exactly 32,000 characters (should pass)
    $max_input = str_repeat('a', 32000);

    $payload = $this->plugin->preparePayload(['input' => $max_input]);

    $this->assertArrayHasKey('input', $payload);
    $this->assertEquals(32000, strlen($payload['input']));
  }

  /**
   * Tests preparePayload includes model parameter.
   */
  public function testPreparePayloadIncludesModel(): void {
    $payload = $this->plugin->preparePayload(['input' => 'Test text']);

    $this->assertArrayHasKey('model', $payload);
  }

}
