<?php

namespace Drupal\api_plugins;

use Drupal\api_plugins\Exception\ApiConfigurationException;

/**
 * Abstract base class for AI API plugins.
 *
 * Provides common functionality shared between AI services like OpenAI,
 * Anthropic, local models, etc. This includes model management,
 * temperature settings, and common AI-specific methods.
 *
 * @package Drupal\api_plugins
 */
abstract class AiApiPluginBase extends ApiPluginBase {

  /**
   * The AI model to use.
   *
   * @var string
   */
  protected string $model = '';

  /**
   * The temperature setting for responses.
   *
   * @var float
   */
  protected float $temperature = 0.0;

  /**
   * Maximum tokens for the response.
   *
   * @var int|null
   */
  protected ?int $maxTokens = NULL;

  /**
   * Top-p sampling parameter.
   *
   * @var float|null
   */
  protected ?float $topP = NULL;

  /**
   * Frequency penalty parameter.
   *
   * @var float|null
   */
  protected ?float $frequencyPenalty = NULL;

  /**
   * Presence penalty parameter.
   *
   * @var float|null
   */
  protected ?float $presencePenalty = NULL;

  /**
   * The HTTP method to use for requests.
   *
   * @var string
   */
  protected string $httpMethod = 'POST';

  /**
   * System prompt for chat models.
   *
   * @var string
   */
  protected string $systemPrompt = 'You are a helpful assistant.';

  /**
   * Get the AI model.
   *
   * @return string
   *   The current model name.
   */
  public function getModel(): string {
    return $this->model;
  }

  /**
   * Set the AI model.
   *
   * @param string $model
   *   The model to use.
   *
   * @return $this
   *   Returns self for method chaining.
   *
   * @throws \InvalidArgumentException
   *   If model name format is invalid.
   */
  public function setModel(string $model): static {
    // Validate model name format (alphanumeric, dots, hyphens, underscores).
    if (!preg_match('/^[a-zA-Z0-9._-]+$/', $model)) {
      throw new \InvalidArgumentException(
        sprintf('Invalid model name format: %s. Model names must contain only alphanumeric characters, dots, hyphens, and underscores.', $model)
      );
    }

    // Set model to check if supported.
    $this->model = $model;

    // Check if model is in supported list (warning only).
    if (!$this->isModelSupported()) {
      try {
        $this->getLogger('api_plugins')->warning(
          'Model @model is not in the list of supported models for @plugin. This may cause unexpected behavior.',
          [
            '@model' => $model,
            '@plugin' => $this->getPluginId(),
          ]
        );
      }
      catch (\Throwable $e) {
        // Logger not available in testing context, silently continue.
      }
    }

    return $this;
  }

  /**
   * Get temperature setting.
   *
   * @return float
   *   Temperature value between 0.0 and 2.0.
   */
  public function getTemperature(): float {
    return $this->temperature;
  }

  /**
   * Set the temperature for AI responses.
   *
   * @param float $temperature
   *   Temperature value between 0.0 and 2.0.
   *   Lower values make responses more focused and deterministic.
   *   Higher values make responses more random and creative.
   *
   * @return $this
   *   Returns self for method chaining.
   */
  public function setTemperature(float $temperature): static {
    $this->temperature = max(0.0, min(2.0, $temperature));
    return $this;
  }

  /**
   * Get maximum tokens setting.
   *
   * @return int|null
   *   Maximum tokens or null if not set.
   */
  public function getMaxTokens(): ?int {
    return $this->maxTokens;
  }

  /**
   * Set maximum tokens for the response.
   *
   * @param int|null $maxTokens
   *   Maximum number of tokens to generate.
   *
   * @return $this
   *   Returns self for method chaining.
   */
  public function setMaxTokens(?int $maxTokens): static {
    $this->maxTokens = $maxTokens;
    return $this;
  }

  /**
   * Get top-p sampling parameter.
   *
   * @return float|null
   *   Top-p value or null if not set.
   */
  public function getTopP(): ?float {
    return $this->topP;
  }

  /**
   * Set top-p sampling parameter.
   *
   * @param float|null $topP
   *   Nucleus sampling parameter (0.0 to 1.0).
   *
   * @return $this
   *   Returns self for method chaining.
   */
  public function setTopP(?float $topP): static {
    if ($topP !== NULL) {
      $this->topP = max(0.0, min(1.0, $topP));
    }
    else {
      $this->topP = NULL;
    }
    return $this;
  }

  /**
   * Get frequency penalty parameter.
   *
   * @return float|null
   *   Frequency penalty value or null if not set.
   */
  public function getFrequencyPenalty(): ?float {
    return $this->frequencyPenalty;
  }

  /**
   * Set frequency penalty parameter.
   *
   * @param float|null $frequencyPenalty
   *   Frequency penalty (-2.0 to 2.0).
   *
   * @return $this
   *   Returns self for method chaining.
   */
  public function setFrequencyPenalty(?float $frequencyPenalty): static {
    if ($frequencyPenalty !== NULL) {
      $this->frequencyPenalty = max(-2.0, min(2.0, $frequencyPenalty));
    }
    else {
      $this->frequencyPenalty = NULL;
    }
    return $this;
  }

  /**
   * Get presence penalty parameter.
   *
   * @return float|null
   *   Presence penalty value or null if not set.
   */
  public function getPresencePenalty(): ?float {
    return $this->presencePenalty;
  }

  /**
   * Set presence penalty parameter.
   *
   * @param float|null $presencePenalty
   *   Presence penalty (-2.0 to 2.0).
   *
   * @return $this
   *   Returns self for method chaining.
   */
  public function setPresencePenalty(?float $presencePenalty): static {
    if ($presencePenalty !== NULL) {
      $this->presencePenalty = max(-2.0, min(2.0, $presencePenalty));
    }
    else {
      $this->presencePenalty = NULL;
    }
    return $this;
  }

  /**
   * Get system prompt.
   *
   * @return string
   *   The system prompt.
   */
  public function getSystemPrompt(): string {
    return $this->systemPrompt;
  }

  /**
   * Set system prompt for chat models.
   *
   * @param string $systemPrompt
   *   The system prompt to use.
   *
   * @return $this
   *   Returns self for method chaining.
   */
  public function setSystemPrompt(string $systemPrompt): static {
    $this->systemPrompt = $systemPrompt;
    return $this;
  }

  /**
   * Get processed system prompt with tokens replaced.
   *
   * @param array $params
   *   Parameters containing token data and options.
   *
   * @return string
   *   The processed system prompt.
   */
  public function getProcessedSystemPrompt(array $params = []): string {
    $system_prompt = $params['system_prompt'] ?? $this->systemPrompt;

    if (!empty($system_prompt) && is_string($system_prompt)) {
      $data = $params['token_data'] ?? [];

      // Enforce safe token options to prevent injection.
      $options = $params['token_options'] ?? [];
      $options['sanitize'] = $options['sanitize'] ?? TRUE;
      $options['clear'] = $options['clear'] ?? TRUE;

      $processed = $this->tokenService->replace($system_prompt, $data, $options);

      // Additional sanitization for AI context to prevent prompt injection.
      $processed = $this->sanitizeForAiContext($processed);

      return $processed;
    }

    return $this->systemPrompt;
  }

  /**
   * Sanitizes text for AI context to prevent prompt injection attacks.
   *
   * Detects and neutralizes common prompt injection patterns that could
   * manipulate the AI's behavior or extract sensitive information.
   *
   * @param string $text
   *   The text to sanitize.
   *
   * @return string
   *   The sanitized text.
   */
  protected function sanitizeForAiContext(string $text): string {
    // Define patterns that indicate prompt injection attempts.
    $injection_patterns = [
      // Command patterns.
      '/\b(ignore|disregard|forget)\s+(all\s+)?(previous|prior|above)\s+(instructions?|commands?|rules?|prompts?)\b/i',
      '/\bnew\s+(instructions?|commands?|rules?|prompts?)\s*:/i',
      '/\bsystem\s*:\s*/i',
      '/\brole\s*:\s*system\b/i',
      '/\bassistant\s*:\s*/i',
      // Data extraction attempts.
      '/\b(print|output|display|show|reveal|return)\s+(your\s+)?(system\s+)?(prompt|instructions|rules|configuration)\b/i',
      '/\bwhat\s+(are\s+)?(your\s+)?(system\s+)?(instructions|rules|prompts)\b/i',
      // Role manipulation attempts.
      '/\byou\s+are\s+now\s+/i',
      '/\bact\s+as\s+(a\s+)?(?!assistant\b)/i',
      '/\bpretend\s+(you|to)\s+/i',
    ];

    // Check for injection patterns and neutralize them.
    $has_injection = FALSE;
    foreach ($injection_patterns as $pattern) {
      if (preg_match($pattern, $text)) {
        $has_injection = TRUE;
        // Replace the matched pattern with a safe placeholder.
        $text = preg_replace($pattern, '[content filtered]', $text);
      }
    }

    // Log if injection attempt was detected.
    if ($has_injection) {
      try {
        $this->getLogger('api_plugins')->warning(
          'Potential prompt injection detected and filtered in plugin @plugin',
          ['@plugin' => $this->getPluginId()]
        );
      }
      catch (\Throwable $e) {
        // Logger not available in testing context, silently continue.
      }
    }

    // Additional safety: limit excessive repetition which can be used
    // for jailbreaking.
    $text = preg_replace('/(.{20,}?)\1{3,}/s', '$1$1', $text);

    return $text;
  }

  /**
   * Get available models for this AI service.
   *
   * Each AI plugin should implement this to return their specific models.
   *
   * @return array
   *   Array of available model names.
   */
  abstract public function listModels(): array;

  /**
   * Validate that the current model is supported.
   *
   * @return bool
   *   TRUE if the model is supported, FALSE otherwise.
   */
  public function isModelSupported(): bool {
    $available_models = $this->listModels();
    return in_array($this->model, $available_models, TRUE);
  }

  /**
   * Get AI-specific configuration parameters.
   *
   * Returns common AI parameters that can be used in payload preparation.
   *
   * @return array
   *   Array of AI configuration parameters.
   */
  protected function getAiParameters(): array {
    $params = [
      'model' => $this->getModel(),
      'temperature' => $this->getTemperature(),
    ];

    if ($this->maxTokens !== NULL) {
      $params['max_tokens'] = $this->maxTokens;
    }

    if ($this->topP !== NULL) {
      $params['top_p'] = $this->topP;
    }

    if ($this->frequencyPenalty !== NULL) {
      $params['frequency_penalty'] = $this->frequencyPenalty;
    }

    if ($this->presencePenalty !== NULL) {
      $params['presence_penalty'] = $this->presencePenalty;
    }

    return $params;
  }

  /**
   * Create a standardized message format for chat models.
   *
   * @param string $role
   *   Message role (system, user, assistant).
   * @param string $content
   *   Message content.
   *
   * @return array
   *   Formatted message array.
   */
  protected function createMessage(string $role, string $content): array {
    return [
      'role' => $role,
      'content' => $content,
    ];
  }

  /**
   * Create messages array for chat completions.
   *
   * @param array $params
   *   Parameters containing prompt and other data.
   *
   * @return array
   *   Array of messages for the API.
   */
  protected function createMessages(array $params = []): array {
    $messages = [];

    $system_prompt = $this->getProcessedSystemPrompt($params);
    if (!empty($system_prompt)) {
      $messages[] = $this->createMessage('system', $system_prompt);
    }

    $prompt = $params['prompt'] ?? '';
    if (!empty($prompt)) {
      $messages[] = $this->createMessage('user', $prompt);
    }

    if (!empty($params['messages']) && is_array($params['messages'])) {
      $user_message = array_pop($messages);
      $messages = array_merge($messages, $params['messages'], [$user_message]);
    }

    return $messages;
  }

  /**
   * Set multiple AI parameters at once.
   *
   * @param array $params
   *   Array of parameters to set.
   *
   * @return $this
   *   Returns self for method chaining.
   */
  public function setAiParameters(array $params): static {
    if (isset($params['model'])) {
      $this->setModel($params['model']);
    }

    if (isset($params['temperature'])) {
      $this->setTemperature($params['temperature']);
    }

    if (isset($params['max_tokens'])) {
      $this->setMaxTokens($params['max_tokens']);
    }

    if (isset($params['top_p'])) {
      $this->setTopP($params['top_p']);
    }

    if (isset($params['frequency_penalty'])) {
      $this->setFrequencyPenalty($params['frequency_penalty']);
    }

    if (isset($params['presence_penalty'])) {
      $this->setPresencePenalty($params['presence_penalty']);
    }

    if (isset($params['system_prompt'])) {
      $this->setSystemPrompt($params['system_prompt']);
    }

    return $this;
  }

  /**
   * Get all current AI parameters.
   *
   * @return array
   *   Array of all current AI parameters.
   */
  public function getAllAiParameters(): array {
    return [
      'model' => $this->getModel(),
      'temperature' => $this->getTemperature(),
      'max_tokens' => $this->getMaxTokens(),
      'top_p' => $this->getTopP(),
      'frequency_penalty' => $this->getFrequencyPenalty(),
      'presence_penalty' => $this->getPresencePenalty(),
      'system_prompt' => $this->getSystemPrompt(),
    ];
  }

  /**
   * {@inheritdoc}
   *
   * Consolidated implementation of getHeaders() for all AI plugins.
   * Uses template method pattern to allow vendor-specific customization.
   *
   * @throws \Drupal\api_plugins\Exception\ApiConfigurationException
   *   If API key is not configured or invalid.
   */
  final public function getHeaders(): array {
    $api_key = $this->getProviderAuthentication();

    if (empty($api_key)) {
      $this->getLogger('api_plugins')->error(
        'API authentication failed for @plugin_id: API key is not configured or invalid.',
        ['@plugin_id' => $this->getPluginId()]
      );
      throw new ApiConfigurationException(
        'API authentication is not properly configured. Please contact the site administrator.',
        $this->getPluginId()
      );
    }

    return array_merge(
      $this->getDefaultHeaders($api_key),
      $this->getVendorSpecificHeaders()
    );
  }

  /**
   * Get default headers common to all AI providers.
   *
   * @param string $api_key
   *   The API authentication key.
   *
   * @return array
   *   Array of default HTTP headers.
   */
  protected function getDefaultHeaders(string $api_key): array {
    return [
      'Authorization' => $api_key,
      'Content-Type' => 'application/json',
    ];
  }

  /**
   * Get vendor-specific headers.
   *
   * Override this method in child classes to add provider-specific headers.
   * For example, Anthropic requires 'anthropic-version' header.
   *
   * @return array
   *   Array of vendor-specific HTTP headers.
   */
  protected function getVendorSpecificHeaders(): array {
    return [];
  }

  /**
   * {@inheritdoc}
   *
   * Consolidated implementation of getAuthentication() for all AI plugins.
   */
  public function getAuthentication(string $plugin_id): string {
    return $this->getProviderAuthentication();
  }

}
