<?php

namespace Drupal\api_plugins;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;

/**
 * Service for managing API authentication by provider.
 *
 * This service uses the Key module for secure API key management.
 * Falls back to environment variables if Key module is not available.
 *
 * Provider authentication configurations are registered via hooks, making
 * the service fully extensible without hardcoded values.
 *
 * @see hook_api_plugins_authentication_info()
 * @see hook_api_plugins_authentication_info_alter()
 */
class ApiAuthenticationService {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The API configuration.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $apiConfig;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The key repository service (optional, from key module).
   *
   * @var \Drupal\key\KeyRepositoryInterface|null
   */
  protected $keyRepository;

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Provider authentication configurations (cached).
   *
   * @var array<string, mixed>|null
   */
  protected $providerConfigs = NULL;

  /**
   * Constructs an ApiAuthenticationService object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   * @param mixed $key_repository
   *   The key repository service (optional, from key module).
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    LoggerChannelFactoryInterface $logger_factory,
    ModuleHandlerInterface $module_handler,
    $key_repository = NULL,
  ) {
    $this->configFactory = $config_factory;
    $this->loggerFactory = $logger_factory;
    $this->moduleHandler = $module_handler;
    $this->keyRepository = $key_repository;
    $this->apiConfig = $config_factory->get('api_plugins.settings');
  }

  /**
   * Get authentication string for a provider.
   *
   * Uses Key module if available, falls back to environment variables.
   *
   * @param string $provider
   *   The provider name (e.g., 'openai', 'anthropic', 'apify').
   *
   * @return string
   *   The authentication string or empty string if none required.
   */
  public function getAuthentication(string $provider): string {
    $configs = $this->getProviderConfigs();

    if (!isset($configs[$provider])) {
      return '';
    }

    $config = $configs[$provider];

    if ($config['auth_type'] === 'none') {
      return '';
    }

    $api_key = $this->getApiKey($provider);

    if (empty($api_key)) {
      $this->loggerFactory->get('api_plugins')
        ->warning('API key not found for provider: @provider', [
          '@provider' => $provider,
        ]);
      return '';
    }

    switch ($config['auth_type']) {
      case 'bearer':
        return 'Bearer ' . $api_key;

      case 'custom':
        return $api_key;

      default:
        return $api_key;
    }
  }

  /**
   * Get API key for a provider.
   *
   * Priority:
   * 1. Key module (if available and configured)
   * 2. Environment variables (fallback)
   *
   * @param string $provider
   *   The provider name.
   *
   * @return string
   *   The API key or empty string if not found.
   */
  protected function getApiKey(string $provider): string {
    $configs = $this->getProviderConfigs();

    if (!isset($configs[$provider])) {
      return '';
    }

    $config = $configs[$provider];

    if ($this->keyRepository && isset($config['config_key'])) {
      $key_id = $this->apiConfig->get($config['config_key']);
      if ($key_id) {
        try {
          $key = $this->keyRepository->getKey($key_id);
          if ($key) {
            $key_value = $key->getKeyValue();
            if (!empty($key_value)) {
              return $key_value;
            }
          }
        }
        catch (\Throwable $e) {
          $this->loggerFactory->get('api_plugins')
            ->error('Error loading key for @provider: @error', [
              '@provider' => $provider,
              '@error' => $e->getMessage(),
            ]);
        }
      }
    }

    if (isset($config['env_var'])) {
      $env_key = getenv($config['env_var']);
      if (empty($env_key) && isset($_ENV[$config['env_var']])) {
        $env_key = $_ENV[$config['env_var']];
      }

      if (!empty($env_key)) {
        return $this->sanitizeApiKey($env_key);
      }
    }

    return '';
  }

  /**
   * Sanitizes an API key to prevent header injection attacks.
   *
   * Removes control characters and validates key format.
   *
   * @param string $key
   *   The API key to sanitize.
   *
   * @return string
   *   The sanitized API key.
   */
  protected function sanitizeApiKey(string $key): string {
    // Remove any whitespace, newlines, or control characters.
    $key = preg_replace('/[\r\n\t\0\x0B\x00-\x1F\x7F]+/', '', $key);
    $key = trim($key);

    // Validate key length (most API keys are 20-500 chars).
    if (strlen($key) < 10 || strlen($key) > 500) {
      $this->loggerFactory->get('api_plugins')
        ->warning('API key has suspicious length: @length characters', [
          '@length' => strlen($key),
        ]);
    }

    // Additional validation: ensure key doesn't contain suspicious patterns.
    if (preg_match('/[<>"\']/', $key)) {
      $this->loggerFactory->get('api_plugins')
        ->warning('API key contains suspicious characters');
    }

    return $key;
  }

  /**
   * Get authentication headers for a provider.
   *
   * @param string $provider
   *   The provider name.
   *
   * @return array
   *   Array of HTTP headers for authentication.
   */
  public function getHeaders(string $provider): array {
    $configs = $this->getProviderConfigs();

    if (!isset($configs[$provider])) {
      return [];
    }

    $config = $configs[$provider];

    if ($config['auth_type'] === 'none') {
      return [];
    }

    $auth = $this->getAuthentication($provider);
    if (empty($auth)) {
      return [];
    }

    $header_name = $config['header_name'] ?? 'Authorization';

    return [
      $header_name => $auth,
    ];
  }

  /**
   * Check if a provider has valid authentication configured.
   *
   * @param string $provider
   *   The provider name.
   *
   * @return bool
   *   TRUE if authentication is configured, FALSE otherwise.
   */
  public function hasAuthentication(string $provider): bool {
    $configs = $this->getProviderConfigs();

    if (!isset($configs[$provider])) {
      return FALSE;
    }

    $config = $configs[$provider];

    if ($config['auth_type'] === 'none') {
      return TRUE;
    }

    $api_key = $this->getApiKey($provider);
    return !empty($api_key);
  }

  /**
   * Get all configured providers with authentication status.
   *
   * @return array
   *   Array of providers with their authentication status.
   */
  public function getProviderStatus(): array {
    $configs = $this->getProviderConfigs();
    $status = [];

    foreach ($configs as $provider => $config) {
      $status[$provider] = [
        'name' => ucfirst($provider),
        'auth_type' => $config['auth_type'],
        'configured' => $this->hasAuthentication($provider),
        'source' => $this->getKeySource($provider),
      ];
    }

    return $status;
  }

  /**
   * Get the source of the API key (Key module or environment variable).
   *
   * @param string $provider
   *   The provider name.
   *
   * @return string
   *   The source: 'key_module', 'environment', or 'none'.
   */
  protected function getKeySource(string $provider): string {
    $configs = $this->getProviderConfigs();

    if (!isset($configs[$provider])) {
      return 'none';
    }

    $config = $configs[$provider];

    if ($config['auth_type'] === 'none') {
      return 'none';
    }

    if ($this->keyRepository && isset($config['config_key'])) {
      $key_id = $this->apiConfig->get($config['config_key']);
      if ($key_id) {
        try {
          $key = $this->keyRepository->getKey($key_id);
          if ($key) {
            $key_value = $key->getKeyValue();
            if (!empty($key_value)) {
              return 'key_module';
            }
          }
        }
        catch (\Throwable $e) {
          $this->loggerFactory->get('api_plugins')
            ->debug('Failed to get key from Key module for @provider: @error', [
              '@provider' => $provider,
              '@error' => $e->getMessage(),
            ]);
        }
      }
    }

    if (isset($config['env_var'])) {
      $env_key = getenv($config['env_var']);
      if (!empty($env_key) || (isset($_ENV[$config['env_var']]) && !empty($_ENV[$config['env_var']]))) {
        return 'environment';
      }
    }

    return 'none';
  }

  /**
   * Get provider authentication configurations from hooks.
   *
   * Invokes hook_api_plugins_authentication_info() and
   * hook_api_plugins_authentication_info_alter().
   *
   * Results are cached in the object for performance.
   *
   * @return array
   *   Array of provider configurations keyed by provider machine name.
   */
  protected function getProviderConfigs(): array {
    if ($this->providerConfigs !== NULL) {
      return $this->providerConfigs;
    }

    $configs = $this->moduleHandler->invokeAll('api_plugins_authentication_info');
    $this->moduleHandler->alter('api_plugins_authentication_info', $configs);
    $this->providerConfigs = $configs;

    return $this->providerConfigs;
  }

}
