<?php

namespace Drupal\api_plugins;

use Drupal\Component\Plugin\PluginBase;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Utility\Token;

/**
 * Abstract base class for API plugins.
 *
 * Provides common functionality for any API endpoint (OpenAI, Ollama,
 * AWS Bedrock,XML/JSON APIs, etc.). Concrete plugins should implement
 * vendor-specific logic.
 *
 * @package Drupal\api_plugins
 */
abstract class ApiPluginBase extends PluginBase implements ContainerFactoryPluginInterface, ApiPluginInterface {

  use StringTranslationTrait;
  use LoggerChannelTrait;

  /**
   * {@inheritdoc}
   *
   * @var array<string, mixed>
   */
  protected $pluginDefinition;

  /**
   * The language manager service.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The API Plugins configuration.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $apiConfig;

  /**
   * The endpoint URL.
   *
   * @var string
   */
  protected string $endpointUrl = '';

  /**
   * The HTTP method to use for requests.
   *
   * @var string
   */
  protected string $httpMethod = 'GET';

  /**
   * The token service.
   *
   * @var \Drupal\Core\Utility\Token
   */
  protected $tokenService;

  /**
   * The authentication service.
   *
   * @var \Drupal\api_plugins\ApiAuthenticationService
   */
  protected $authenticationService;

  /**
   * {@inheritdoc}
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    LanguageManagerInterface $language_manager,
    ConfigFactoryInterface $config_factory,
    Token $token_service,
    ApiAuthenticationService $authentication_service,
  ) {
    $this->pluginDefinition = $plugin_definition;
    $this->languageManager = $language_manager;
    $this->configFactory = $config_factory;
    $this->apiConfig = $config_factory->get('api_plugins.settings');
    $this->tokenService = $token_service;
    $this->authenticationService = $authentication_service;

    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    // @phpstan-ignore-next-line
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('language_manager'),
      $container->get('config.factory'),
      $container->get('token'),
      $container->get('api_plugins.authentication')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function setEndpoint(string $url): void {
    $this->endpointUrl = $url;
  }

  /**
   * {@inheritdoc}
   */
  public function getEndpoint(): string {
    if (isset($this->endpointUrl)) {
      return $this->endpointUrl;
    }
    return $this->pluginDefinition['endpointUrl'] ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public function getVendor(): string {
    if (!empty($this->pluginDefinition['vendor'])) {
      return $this->pluginDefinition['vendor'];
    }

    if (!empty($this->pluginDefinition['defaultConfig']['vendor'])) {
      return $this->pluginDefinition['defaultConfig']['vendor'];
    }

    return '';
  }

  /**
   * {@inheritdoc}
   */
  abstract public function preparePayload(array $params = []): array;

  /**
   * {@inheritdoc}
   */
  abstract public function getHeaders(): array;

  /**
   * Optional hook invoked by the request service before preparing payload.
   *
   * Plugins may override this to perform per-request preparation such as
   * initializing sessions or mapping parameters. The default implementation
   * does nothing so existing plugins are not forced to implement it.
   *
   * @param array $params
   *   The incoming request parameters passed to the request service.
   */
  public function prepareForRequest(array $params = []): void {
    // Default no-op. Subclasses (for example MCP plugins) can override.
  }

  /**
   * {@inheritdoc}
   */
  abstract public function formatResponse(array $result): string|array;

  /**
   * {@inheritdoc}
   */
  public function validateResponse(array $response): bool {
    if (isset($response['status_code'])) {
      $status_code = $response['status_code'];

      if ($status_code >= 400 && $status_code < 500) {
        $error_message = $response['error_message'] ?? 'Client error occurred';
        throw new \RuntimeException(sprintf(
          'API client error (HTTP %d): %s',
          $status_code,
          $error_message
        ));
      }

      if ($status_code >= 500 && $status_code < 600) {
        $error_message = $response['error_message'] ?? 'Server error occurred';
        throw new \RuntimeException(sprintf(
          'API server error (HTTP %d): %s',
          $status_code,
          $error_message
        ));
      }
    }

    return !empty($response);
  }

  /**
   * {@inheritdoc}
   */
  public function setHttpMethod(string $method): void {
    $valid_methods = ['GET', 'POST', 'PUT', 'DELETE'];
    if (in_array($method, $valid_methods)) {
      $this->httpMethod = $method;
    }
    else {
      throw new \InvalidArgumentException("Invalid HTTP method: $method");
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getHttpMethod(): string {
    return $this->httpMethod;
  }

  /**
   * {@inheritdoc}
   */
  abstract public function getAuthentication(string $plugin_id): string;

  /**
   * Get authentication for the plugin's provider.
   *
   * @return string
   *   Authentication string for this plugin's provider.
   */
  protected function getProviderAuthentication(): string {
    $provider = $this->getVendor();
    return $this->authenticationService->getAuthentication(strtolower($provider));
  }

  /**
   * Get default headers for the plugin's provider.
   *
   * @return array
   *   Default headers including authentication.
   */
  protected function getProviderHeaders(): array {
    $provider = $this->getVendor();
    return $this->authenticationService->getHeaders(strtolower($provider));
  }

  /**
   * Check if the plugin's provider has valid authentication.
   *
   * @return bool
   *   TRUE if authentication is valid or not required.
   */
  protected function isProviderAuthValid(): bool {
    $provider = $this->getVendor();
    return $this->authenticationService->hasAuthentication(strtolower($provider));
  }

  /**
   * Get default configuration from plugin definition.
   *
   * @return array
   *   Default configuration array from @ApiPlugin annotation.
   */
  public function getDefaultConfiguration(): array {
    return $this->pluginDefinition['default_config'] ?? [];
  }

  /**
   * Get plugin configuration.
   *
   * Returns configuration from data source widget if available,
   * otherwise falls back to default configuration from plugin definition.
   *
   * @return array
   *   Plugin configuration array.
   */
  public function getConfiguration(): array {
    $runtime_config = $this->configuration;
    $plugin_defaults = $this->getDefaultConfiguration();
    $merged_config = array_merge($plugin_defaults, $runtime_config);

    return $merged_config;
  }

}
