<?php

namespace Drupal\api_plugins;

/**
 * Defines the interface for API plugins.
 *
 * This interface provides a common contract for all API service providers
 * to implement HTTP-based API interactions. Can be used for AI services,
 * payment gateways, social media APIs, weather services, etc.
 *
 * @see \Drupal\api_plugins\AiEndpointManager
 * @see \Drupal\api_plugins\Annotation\AiPlugin
 * @see plugin_api
 */
interface ApiPluginInterface {

  /**
   * Prepare the API request payload.
   *
   * @param array $params
   *   Request parameters to include in the payload.
   *
   * @return array
   *   The prepared payload array for the API request.
   *
   * @throws \InvalidArgumentException
   *   When data is empty or invalid parameters are provided.
   */
  public function preparePayload(array $params = []): array;

  /**
   * Get the HTTP headers for the API request.
   *
   * @return array
   *   An associative array of HTTP headers.
   *
   * @throws \Drupal\api_plugins\Exception\ApiConfigurationException
   *   When API key or authentication is not properly configured.
   */
  public function getHeaders(): array;

  /**
   * Get the API endpoint URL.
   *
   * @return string
   *   The full API endpoint URL.
   */
  public function getEndpoint(): string;

  /**
   * Set the API endpoint URL.
   *
   * @param string $url
   *   The API endpoint URL.
   *
   * @throws \InvalidArgumentException
   *   When the URL is invalid or malformed.
   */
  public function setEndpoint(string $url): void;

  /**
   * Get the API service vendor/provider name.
   *
   * @return string
   *   The vendor name (e.g., 'openai', 'stripe', 'mailchimp').
   */
  public function getVendor(): string;

  /**
   * Format the API response into a standardized format.
   *
   * @param array $result
   *   The raw API response.
   *
   * @return string|array
   *   The formatted response data.
   *
   * @throws \Drupal\api_plugins\Exception\ApiResponseException
   *   When the response cannot be parsed or is invalid.
   */
  public function formatResponse(array $result): string|array;

  /**
   * Validate the API response structure.
   *
   * @param array $response
   *   The API response to validate.
   *
   * @return bool
   *   TRUE if the response is valid, FALSE otherwise.
   */
  public function validateResponse(array $response): bool;

  /**
   * Set the HTTP method for the API request.
   *
   * @param string $method
   *   The HTTP method (e.g., 'GET', 'POST', 'PUT', 'DELETE').
   *
   * @throws \InvalidArgumentException
   *   When the method is not a valid HTTP method.
   */
  public function setHttpMethod(string $method): void;

  /**
   * Get the HTTP method for the API request.
   *
   * @return string
   *   The HTTP method (e.g., 'GET', 'POST', 'PUT', 'DELETE').
   */
  public function getHttpMethod(): string;

  /**
   * Get authentication data for the AI provider.
   *
   * @return string
   *   Authentication data.
   */
  public function getAuthentication(string $plugin_id): string;

}
