<?php

namespace Drupal\api_plugins;

use Drupal\api_plugins\Exception\ApiAuthenticationException;
use Drupal\api_plugins\Exception\ApiConnectionException;
use Drupal\api_plugins\Exception\ApiRateLimitException;
use Drupal\api_plugins\Exception\ApiResponseException;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Config\ConfigFactory;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Logger\LoggerChannelTrait;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;
use GuzzleHttp\Exception\RequestException;

/**
 * API Request Service.
 *
 * Handles HTTP requests to API plugins and manages plugin lifecycle.
 *
 * @package Drupal\api_plugins
 */
class ApiRequestService {

  use LoggerChannelTrait;

  /**
   * ChatGPT  language model.
   *
   * @var string
   */
  protected $model;

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The API plugin discovery manager.
   *
   * @var \Drupal\api_plugins\ApiPluginDiscovery
   */
  protected $pluginManager;

  /**
   * The API configuration.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $apiConfig;

  /**
   * A guzzle http client instance.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * {@inheritdoc}
   */
  public function __construct(ConfigFactory $config_factory, ClientInterface $http_client, ApiPluginDiscovery $plugin_manager, ModuleHandlerInterface $module_handler) {
    $this->configFactory = $config_factory;
    $this->httpClient = $http_client;
    $this->apiConfig = $this->configFactory->get('api_plugins.settings');
    $this->pluginManager = $plugin_manager;
    $this->moduleHandler = $module_handler;
  }

  /**
   * Get API module configuration.
   *
   * @return \Drupal\Core\Config\Config|\Drupal\Core\Config\ImmutableConfig
   *   API configuration.
   */
  public function getConfig() {
    return $this->configFactory->get('api_plugins.settings');
  }

  /**
   * Send API request using specified plugin.
   *
   * @param string $plugin_id
   *   The plugin ID to use for the request.
   * @param array $params
   *   Parameters to include in the request.
   *
   * @return mixed
   *   The formatted response from the API.
   *
   * @throws \Exception
   *   Throws exception on request failure.
   */
  public function sendRequest(string $plugin_id, array $params) {
    $plugin = $this->pluginManager->createInstance($plugin_id);
    if (!$plugin instanceof ApiPluginBase) {
      throw new \InvalidArgumentException(sprintf('Plugin %s is not an instance of ApiPluginBase.', $plugin_id));
    }
    // Allow plugin to prepare for the request (session init, param mapping,
    // etc.) without ApiRequestService needing to know plugin subtypes.
    try {
      $plugin->prepareForRequest($params);
    }
    catch (\Exception $e) {
      throw new \RuntimeException('Plugin preparation failed for ' . $plugin_id . ': ' . $e->getMessage());
    }

    $payload = $plugin->preparePayload($params);

    $url = $plugin->getEndpoint();
    $this->getLogger('api_plugins')->error('Plugin ID: @id, Endpoint URL: @url, Class: @class',
    ['@id' => $plugin_id, '@url' => $url, '@class' => get_class($plugin)]);

    $context = [
      'plugin_id' => $plugin_id,
      'plugin' => $plugin,
      'params' => $params,
      'endpoint' => $url,
    ];
    $this->moduleHandler->alter('api_plugins_prepare_payload', $payload, $context);

    // Get and validate timeout configuration.
    $timeout = $this->getValidatedTimeout($params);
    $connect_timeout = $this->getValidatedConnectTimeout($params);

    $options = [
      'headers' => $plugin->getHeaders(),
      'json' => $payload,
      'timeout' => $timeout,
      'connect_timeout' => $connect_timeout,
    ];

    $decoded_data = NULL;

    try {
      $response = $this->httpClient->request($plugin->getHttpMethod(), $url, $options);
      $status_code = $response->getStatusCode();
      $result = $response->getBody()->getContents();
      $decoded_data = Json::decode($result);

      $response_with_status = [
        'status_code' => $status_code,
        'data' => $decoded_data,
      ];

      $plugin->validateResponse($response_with_status);
      $decoded_data = $plugin->formatResponse($decoded_data);
    }
    catch (RequestException $exception) {
      $status_code = NULL;
      $detailed_error = $exception->getMessage();

      if ($exception->hasResponse()) {
        $response = $exception->getResponse();
        $status_code = $response->getStatusCode();
        $error_body = $response->getBody()->getContents();

        try {
          $error_data = Json::decode($error_body);
          $detailed_error = $error_data['error']['message'] ?? $error_data['message'] ?? $exception->getMessage();
        }
        catch (\Exception $e) {
          $detailed_error = $exception->getMessage();
        }

        // Log detailed error for administrators.
        $this->getLogger('api_plugins')->error('API request failed for @plugin: @error', [
          '@plugin' => $plugin_id,
          '@error' => $detailed_error,
          'status_code' => $status_code,
          'endpoint' => $plugin->getEndpoint(),
        ]);

        // Throw appropriate exception with safe user message.
        $this->handleApiError($status_code, $plugin->getEndpoint(), $detailed_error);
      }

      // No response - connection error.
      $this->getLogger('api_plugins')->error('API connection failed for @plugin: @error', [
        '@plugin' => $plugin_id,
        '@error' => $detailed_error,
      ]);

      throw new ApiConnectionException(
        'API service is currently unavailable. Please try again later.',
        $plugin->getEndpoint()
      );
    }
    catch (GuzzleException $exception) {
      $this->getLogger('api_plugins')->error('API request exception for @plugin: @error', [
        '@plugin' => $plugin_id,
        '@error' => $exception->getMessage(),
      ]);

      throw new ApiConnectionException(
        'API request failed. Please contact the administrator.',
        $plugin->getEndpoint()
      );
    }
    return $decoded_data;
  }

  /**
   * Handles API errors by throwing appropriate exceptions with safe messages.
   *
   * @param int $status_code
   *   The HTTP status code.
   * @param string $endpoint
   *   The API endpoint.
   * @param string $detailed_error
   *   The detailed error message (for logging only).
   *
   * @throws \Drupal\api_plugins\Exception\ApiAuthenticationException
   *   For authentication errors (401, 403).
   * @throws \Drupal\api_plugins\Exception\ApiRateLimitException
   *   For rate limit errors (429).
   * @throws \Drupal\api_plugins\Exception\ApiResponseException
   *   For other API errors.
   */
  protected function handleApiError(int $status_code, string $endpoint, string $detailed_error): void {
    $safe_messages = [
      400 => 'Invalid request. Please check your input.',
      401 => 'Authentication failed. Please check your API configuration.',
      403 => 'Access denied. Please contact the administrator.',
      404 => 'API endpoint not found.',
      429 => 'Too many requests. Please try again later.',
      500 => 'API service error. Please try again later.',
      502 => 'API gateway error. Please try again later.',
      503 => 'API service temporarily unavailable.',
      504 => 'API gateway timeout. Please try again later.',
    ];

    $message = $safe_messages[$status_code] ?? 'API request failed. Please try again later.';

    // Throw specific exception types.
    switch ($status_code) {
      case 401:
      case 403:
        throw new ApiAuthenticationException(
          $message,
          '',
          $status_code
        );

      case 429:
        throw new ApiRateLimitException(
          $message,
          60,
          $status_code
        );

      default:
        throw new ApiResponseException(
          $message,
          $status_code
        );
    }
  }

  /**
   * Gets and validates the request timeout value.
   *
   * @param array $params
   *   Request parameters.
   *
   * @return int
   *   Validated timeout value in seconds.
   */
  protected function getValidatedTimeout(array $params): int {
    // Check params first, then config, then default.
    $timeout = $params['timeout'] ?? $this->apiConfig->get('request_timeout') ?? 90;

    // Validate timeout is within acceptable range (5-300 seconds).
    $timeout = (int) $timeout;
    if ($timeout < 5) {
      $this->getLogger('api_plugins')->warning('Request timeout too low (@timeout seconds), using minimum of 5 seconds.', [
        '@timeout' => $timeout,
      ]);
      $timeout = 5;
    }
    elseif ($timeout > 300) {
      $this->getLogger('api_plugins')->warning('Request timeout too high (@timeout seconds), using maximum of 300 seconds.', [
        '@timeout' => $timeout,
      ]);
      $timeout = 300;
    }

    return $timeout;
  }

  /**
   * Gets and validates the connection timeout value.
   *
   * @param array $params
   *   Request parameters.
   *
   * @return int
   *   Validated connect timeout value in seconds.
   */
  protected function getValidatedConnectTimeout(array $params): int {
    // Check params first, then config, then default.
    $connect_timeout = $params['connect_timeout'] ?? $this->apiConfig->get('connect_timeout') ?? 15;

    // Validate connect timeout is within acceptable range (5-60 seconds).
    $connect_timeout = (int) $connect_timeout;
    if ($connect_timeout < 5) {
      $this->getLogger('api_plugins')->warning('Connect timeout too low (@timeout seconds), using minimum of 5 seconds.', [
        '@timeout' => $connect_timeout,
      ]);
      $connect_timeout = 5;
    }
    elseif ($connect_timeout > 60) {
      $this->getLogger('api_plugins')->warning('Connect timeout too high (@timeout seconds), using maximum of 60 seconds.', [
        '@timeout' => $connect_timeout,
      ]);
      $connect_timeout = 60;
    }

    return $connect_timeout;
  }

  /**
   * Get options ready list of available AI Plugins.
   *
   * @return array
   *   Associative array of plugin IDs and titles.
   */
  public function listAiPlugins() {
    $definitions = $this->pluginManager->getDefinitions();
    $options = [];
    foreach ($definitions as $key => $definition) {
      $options[$key] = $definition['title'];
    }
    return $options;
  }

  /**
   * Get options ready list of models for a given endpoint plugin.
   *
   * @param string $endpoint
   *   The endpoint plugin ID.
   *
   * @return array
   *   Associative array of model names.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  public function listEndpointModels(string $endpoint) {
    $plugin = $this->pluginManager->createInstance($endpoint);
    $list = $plugin->listModels();
    $options = [];
    foreach ($list as $option) {
      $options[$option] = $option;
    }
    return $options;
  }

}
