<?php

namespace Drupal\api_plugins\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\key\KeyRepositoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for API Plugins module.
 */
class ApiConfigForm extends ConfigFormBase {

  /**
   * Config settings.
   */
  const CONFIG_NAME = 'api_plugins.settings';

  /**
   * The key repository service.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */
  protected $keyRepository;

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Constructs a new ApiConfigForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\key\KeyRepositoryInterface $key_repository
   *   The key repository service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config_manager
   *   The typed config manager.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    KeyRepositoryInterface $key_repository,
    ModuleHandlerInterface $module_handler,
    TypedConfigManagerInterface $typed_config_manager,
  ) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->keyRepository = $key_repository;
    $this->moduleHandler = $module_handler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('config.factory'),
      $container->get('key.repository'),
      $container->get('module_handler'),
      $container->get('config.typed')
    );
  }

  /**
   * {@inheritdoc}
   *
   * @return array<string>
   *   An array of configuration names.
   */
  protected function getEditableConfigNames(): array {
    return [
      static::CONFIG_NAME,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'api_plugins_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config(static::CONFIG_NAME);

    $form['api_keys'] = [
      '#type' => 'details',
      '#title' => $this->t('API Keys Configuration'),
      '#description' => $this->t('Configure API keys using the Key module for secure storage.'),
      '#open' => TRUE,
    ];

    $form['api_keys']['info'] = [
      '#type' => 'markup',
      '#markup' => '<div class="messages messages--info">' .
      '<p><strong>' . $this->t('Using Drupal Key Module') . '</strong></p>' .
      '<p>' . $this->t('API keys are securely managed through the <a href="@key_url">Key module</a>.', [
        '@key_url' => '/admin/config/system/keys',
      ]) . '</p>' .
      '<p>' . $this->t('Create keys at: <strong>Configuration → System → Keys</strong>') . '</p>' .
      '</div>',
    ];

    $key_options = $this->getKeyOptions();

    if (empty($key_options)) {
      $form['api_keys']['no_keys_warning'] = [
        '#type' => 'markup',
        '#markup' => '<div class="messages messages--warning">' .
        '<p><strong>' . $this->t('No keys found!') . '</strong></p>' .
        '<p>' . $this->t('Please create API keys first at: <a href="@key_url">Configuration → System → Keys</a>', [
          '@key_url' => '/admin/config/system/keys/add',
        ]) . '</p>' .
        '</div>',
      ];
    }

    $api_key_info = $this->getApiKeyInfo();

    foreach ($api_key_info as $key_id => $info) {
      $field_name = $key_id . '_key';
      $config_key = $info['config_key'];

      $form['api_keys'][$field_name] = [
        '#type' => 'select',
        '#title' => $info['label'],
        '#description' => $info['description'],
        '#options' => ['' => $this->t('- None -')] + $key_options,
        '#default_value' => $config->get($config_key) ?: '',
        '#weight' => $info['weight'] ?? 0,
      ];

      $selected_key_id = $config->get($config_key);
      if ($selected_key_id) {
        $selected_key = $this->keyRepository->getKey($selected_key_id);
        if ($selected_key) {
          $form['api_keys'][$field_name . '_status'] = [
            '#type' => 'item',
            '#markup' => '<span style="color: green;">✅ ' . $this->t('Key configured: @name', [
              '@name' => $selected_key->label(),
            ]) . '</span>',
            '#weight' => ($info['weight'] ?? 0) + 0.5,
          ];
        }
        else {
          $form['api_keys'][$field_name . '_status'] = [
            '#type' => 'item',
            '#markup' => '<span style="color: red;">❌ ' . $this->t('Selected key not found!') . '</span>',
            '#weight' => ($info['weight'] ?? 0) + 0.5,
          ];
        }
      }
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->config(static::CONFIG_NAME);
    $api_key_info = $this->getApiKeyInfo();
    foreach ($api_key_info as $key_id => $info) {
      $field_name = $key_id . '_key';
      $config_key = $info['config_key'];
      $value = $form_state->getValue($field_name);

      $config->set($config_key, $value);
    }

    $config->save();

    $this->messenger()->addMessage($this->t('API key configuration saved successfully.'));

    parent::submitForm($form, $form_state);
  }

  /**
   * Get available keys as options for select fields.
   *
   * @return array<string, string>
   *   Array of key options keyed by key ID.
   */
  protected function getKeyOptions(): array {
    $options = [];

    $keys = $this->keyRepository->getKeys();
    foreach ($keys as $key) {
      $options[$key->id()] = $key->label();
    }

    return $options;
  }

  /**
   * Get API key information from all modules via hook.
   *
   * @return array<string, array<string, mixed>>
   *   Array of API key definitions, keyed by unique identifier.
   *   Each definition contains:
   *   - label: The human-readable label.
   *   - description: Help text for the field.
   *   - config_key: The configuration key for storage.
   *   - weight: Display order weight.
   */
  protected function getApiKeyInfo(): array {
    $api_key_info = $this->moduleHandler->invokeAll('api_plugins_api_key_info');
    $this->moduleHandler->alter('api_plugins_api_key_info', $api_key_info);
    uasort($api_key_info, function ($a, $b) {
      $a_weight = $a['weight'] ?? 0;
      $b_weight = $b['weight'] ?? 0;
      return $a_weight <=> $b_weight;
    });

    return $api_key_info;
  }

}
