<?php

namespace Drupal\Tests\api_plugins\Unit;

use Drupal\api_plugins\AiApiPluginBase;
use Drupal\api_plugins\Exception\ApiConfigurationException;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;
use Psr\Log\LoggerInterface;

/**
 * Tests for AiApiPluginBase.
 */
#[CoversClass(AiApiPluginBase::class)]
#[Group('api_plugins')]
class AiApiPluginBaseTest extends UnitTestCase {

  /**
   * The mock AI plugin.
   *
   * @var \Drupal\api_plugins\AiApiPluginBase|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $plugin;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->plugin = $this->getMockBuilder(AiApiPluginBase::class)
      ->disableOriginalConstructor()
      ->onlyMethods([
        'listModels',
        'getProviderAuthentication',
        'getLogger',
        't',
        'getPluginId',
        'preparePayload',
        'formatResponse',
      ])
      ->getMock();
  }

  /**
   * Tests model getter and setter.
   */
  public function testModelGetterSetter(): void {
    $model = 'gpt-4o';

    $result = $this->plugin->setModel($model);
    $this->assertSame($this->plugin, $result, 'setModel should return $this for chaining');
    $this->assertEquals($model, $this->plugin->getModel());
  }

  /**
   * Tests temperature getter and setter.
   */
  public function testTemperatureGetterSetter(): void {
    $temperature = 0.7;

    $result = $this->plugin->setTemperature($temperature);
    $this->assertSame($this->plugin, $result);
    $this->assertEquals($temperature, $this->plugin->getTemperature());
  }

  /**
   * Tests temperature clamping to valid range.
   */
  #[DataProvider('temperatureProvider')]
  public function testTemperatureClamping($input, $expected): void {
    $this->plugin->setTemperature($input);
    $this->assertEquals($expected, $this->plugin->getTemperature());
  }

  /**
   * Data provider for temperature clamping tests.
   */
  public static function temperatureProvider(): array {
    return [
      'below minimum' => [-1.0, 0.0],
      'minimum' => [0.0, 0.0],
      'normal' => [0.7, 0.7],
      'maximum' => [2.0, 2.0],
      'above maximum' => [3.0, 2.0],
    ];
  }

  /**
   * Tests maxTokens getter and setter.
   */
  public function testMaxTokensGetterSetter(): void {
    $maxTokens = 1000;

    $result = $this->plugin->setMaxTokens($maxTokens);
    $this->assertSame($this->plugin, $result);
    $this->assertEquals($maxTokens, $this->plugin->getMaxTokens());
  }

  /**
   * Tests maxTokens can be set to NULL.
   */
  public function testMaxTokensNull(): void {
    $this->plugin->setMaxTokens(1000);
    $this->plugin->setMaxTokens(NULL);
    $this->assertNull($this->plugin->getMaxTokens());
  }

  /**
   * Tests topP getter and setter.
   */
  public function testTopGetterSetter(): void {
    $topP = 0.9;

    $result = $this->plugin->setTopP($topP);
    $this->assertSame($this->plugin, $result);
    $this->assertEquals($topP, $this->plugin->getTopP());
  }

  /**
   * Tests topP clamping to valid range.
   */
  #[DataProvider('topProvider')]
  public function testTopClamping($input, $expected): void {
    $this->plugin->setTopP($input);
    $this->assertEquals($expected, $this->plugin->getTopP());
  }

  /**
   * Data provider for topP clamping tests.
   */
  public static function topProvider(): array {
    return [
      'below minimum' => [-0.5, 0.0],
      'minimum' => [0.0, 0.0],
      'normal' => [0.9, 0.9],
      'maximum' => [1.0, 1.0],
      'above maximum' => [1.5, 1.0],
      'null' => [NULL, NULL],
    ];
  }

  /**
   * Tests frequency penalty getter and setter.
   */
  public function testFrequencyPenaltyGetterSetter(): void {
    $penalty = 0.5;

    $result = $this->plugin->setFrequencyPenalty($penalty);
    $this->assertSame($this->plugin, $result);
    $this->assertEquals($penalty, $this->plugin->getFrequencyPenalty());
  }

  /**
   * Tests frequency penalty clamping.
   */
  #[DataProvider('penaltyProvider')]
  public function testFrequencyPenaltyClamping($input, $expected): void {
    $this->plugin->setFrequencyPenalty($input);
    $this->assertEquals($expected, $this->plugin->getFrequencyPenalty());
  }

  /**
   * Tests presence penalty getter and setter.
   */
  public function testPresencePenaltyGetterSetter(): void {
    $penalty = 0.5;

    $result = $this->plugin->setPresencePenalty($penalty);
    $this->assertSame($this->plugin, $result);
    $this->assertEquals($penalty, $this->plugin->getPresencePenalty());
  }

  /**
   * Tests presence penalty clamping.
   */
  #[DataProvider('penaltyProvider')]
  public function testPresencePenaltyClamping($input, $expected): void {
    $this->plugin->setPresencePenalty($input);
    $this->assertEquals($expected, $this->plugin->getPresencePenalty());
  }

  /**
   * Data provider for penalty clamping tests.
   */
  public static function penaltyProvider(): array {
    return [
      'below minimum' => [-3.0, -2.0],
      'minimum' => [-2.0, -2.0],
      'zero' => [0.0, 0.0],
      'normal' => [0.5, 0.5],
      'maximum' => [2.0, 2.0],
      'above maximum' => [3.0, 2.0],
      'null' => [NULL, NULL],
    ];
  }

  /**
   * Tests system prompt getter and setter.
   */
  public function testSystemPromptGetterSetter(): void {
    $prompt = 'You are a helpful coding assistant.';

    $result = $this->plugin->setSystemPrompt($prompt);
    $this->assertSame($this->plugin, $result);
    $this->assertEquals($prompt, $this->plugin->getSystemPrompt());
  }

  /**
   * Tests createMessage helper method.
   */
  public function testCreateMessage(): void {
    $reflection = new \ReflectionClass($this->plugin);
    $method = $reflection->getMethod('createMessage');
    $method->setAccessible(TRUE);

    $message = $method->invoke($this->plugin, 'user', 'Hello world');

    $this->assertIsArray($message);
    $this->assertArrayHasKey('role', $message);
    $this->assertArrayHasKey('content', $message);
    $this->assertEquals('user', $message['role']);
    $this->assertEquals('Hello world', $message['content']);
  }

  /**
   * Tests getAiParameters method.
   */
  public function testGetAiParameters(): void {
    $this->plugin->setModel('gpt-4');
    $this->plugin->setTemperature(0.7);
    $this->plugin->setMaxTokens(1000);
    $this->plugin->setTopP(0.9);

    $reflection = new \ReflectionClass($this->plugin);
    $method = $reflection->getMethod('getAiParameters');
    $method->setAccessible(TRUE);

    $params = $method->invoke($this->plugin);

    $this->assertIsArray($params);
    $this->assertEquals('gpt-4', $params['model']);
    $this->assertEquals(0.7, $params['temperature']);
    $this->assertEquals(1000, $params['max_tokens']);
    $this->assertEquals(0.9, $params['top_p']);
  }

  /**
   * Tests getAiParameters excludes null values.
   */
  public function testGetAiParametersExcludesNulls(): void {
    $this->plugin->setModel('gpt-4');
    $this->plugin->setTemperature(0.7);
    $this->plugin->setMaxTokens(NULL);

    $reflection = new \ReflectionClass($this->plugin);
    $method = $reflection->getMethod('getAiParameters');
    $method->setAccessible(TRUE);

    $params = $method->invoke($this->plugin);

    $this->assertArrayNotHasKey('max_tokens', $params);
    $this->assertArrayHasKey('model', $params);
    $this->assertArrayHasKey('temperature', $params);
  }

  /**
   * Tests setAiParameters sets multiple parameters at once.
   */
  public function testSetAiParameters(): void {
    $params = [
      'model' => 'gpt-4',
      'temperature' => 0.8,
      'max_tokens' => 2000,
      'top_p' => 0.95,
      'frequency_penalty' => 0.5,
      'presence_penalty' => 0.3,
      'system_prompt' => 'Custom prompt',
    ];

    $result = $this->plugin->setAiParameters($params);

    $this->assertSame($this->plugin, $result);
    $this->assertEquals('gpt-4', $this->plugin->getModel());
    $this->assertEquals(0.8, $this->plugin->getTemperature());
    $this->assertEquals(2000, $this->plugin->getMaxTokens());
    $this->assertEquals(0.95, $this->plugin->getTopP());
    $this->assertEquals(0.5, $this->plugin->getFrequencyPenalty());
    $this->assertEquals(0.3, $this->plugin->getPresencePenalty());
    $this->assertEquals('Custom prompt', $this->plugin->getSystemPrompt());
  }

  /**
   * Tests getAllAiParameters returns all current values.
   */
  public function testGetAllAiParameters(): void {
    $this->plugin->setModel('claude-3');
    $this->plugin->setTemperature(0.5);
    $this->plugin->setMaxTokens(1500);

    $params = $this->plugin->getAllAiParameters();

    $this->assertIsArray($params);
    $this->assertEquals('claude-3', $params['model']);
    $this->assertEquals(0.5, $params['temperature']);
    $this->assertEquals(1500, $params['max_tokens']);
  }

  /**
   * Tests getHeaders with valid authentication.
   */
  public function testGetHeadersWithValidAuth(): void {
    $this->plugin->method('getProviderAuthentication')
      ->willReturn('Bearer test_key_123');

    $this->plugin->method('getPluginId')
      ->willReturn('test_plugin');

    $headers = $this->plugin->getHeaders();

    $this->assertIsArray($headers);
    $this->assertArrayHasKey('Authorization', $headers);
    $this->assertArrayHasKey('Content-Type', $headers);
    $this->assertEquals('Bearer test_key_123', $headers['Authorization']);
    $this->assertEquals('application/json', $headers['Content-Type']);
  }

  /**
   * Tests getHeaders throws exception with empty authentication.
   */
  public function testGetHeadersThrowsExceptionWithEmptyAuth(): void {
    $this->plugin->method('getProviderAuthentication')
      ->willReturn('');

    $this->plugin->method('getPluginId')
      ->willReturn('test_plugin');

    $logger = $this->createMock(LoggerInterface::class);
    $logger->expects($this->once())
      ->method('error');

    $this->plugin->method('getLogger')
      ->willReturn($logger);

    $this->plugin->method('t')
      ->willReturn('API authentication is not properly configured');

    $this->expectException(ApiConfigurationException::class);
    $this->expectExceptionMessage('API authentication is not properly configured');

    $this->plugin->getHeaders();
  }

  /**
   * Tests isModelSupported returns TRUE for valid model.
   */
  public function testIsModelSupportedTrue(): void {
    $this->plugin->method('listModels')
      ->willReturn(['gpt-4', 'gpt-4-turbo', 'gpt-3.5-turbo']);

    $this->plugin->setModel('gpt-4');

    $this->assertTrue($this->plugin->isModelSupported());
  }

  /**
   * Tests isModelSupported returns FALSE for invalid model.
   */
  public function testIsModelSupportedFalse(): void {
    $this->plugin->method('listModels')
      ->willReturn(['gpt-4', 'gpt-4-turbo']);

    $this->plugin->setModel('invalid-model');

    $this->assertFalse($this->plugin->isModelSupported());
  }

  /**
   * Tests getAuthentication delegates to getProviderAuthentication.
   */
  public function testGetAuthentication(): void {
    $expected_auth = 'Bearer test_key';

    $this->plugin->method('getProviderAuthentication')
      ->willReturn($expected_auth);

    $result = $this->plugin->getAuthentication('test_plugin');

    $this->assertEquals($expected_auth, $result);
  }

}
