<?php

namespace Drupal\Tests\api_plugins\Unit\Exception;

use Drupal\api_plugins\Exception\ApiConfigurationException;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests for ApiConfigurationException.
 */
#[CoversClass(ApiConfigurationException::class)]
#[Group('api_plugins')]
class ApiConfigurationExceptionTest extends UnitTestCase {

  /**
   * Tests basic exception creation.
   */
  public function testBasicExceptionCreation(): void {
    $message = 'Test configuration error';
    $plugin_id = 'test_plugin';
    $code = 500;

    $exception = new ApiConfigurationException($message, $plugin_id, $code);

    $this->assertEquals($message, $exception->getMessage());
    $this->assertEquals($plugin_id, $exception->getPluginId());
    $this->assertEquals($code, $exception->getCode());
  }

  /**
   * Tests exception without plugin ID.
   */
  public function testExceptionWithoutPluginId(): void {
    $message = 'Test error without plugin ID';
    $exception = new ApiConfigurationException($message);

    $this->assertEquals($message, $exception->getMessage());
    $this->assertNull($exception->getPluginId());
    $this->assertEquals(0, $exception->getCode());
  }

  /**
   * Tests exception with previous exception.
   */
  public function testExceptionWithPrevious(): void {
    $previous = new \RuntimeException('Previous error');
    $exception = new ApiConfigurationException('Current error', 'test_plugin', 0, $previous);

    $this->assertSame($previous, $exception->getPrevious());
  }

  /**
   * Tests that exception extends RuntimeException.
   */
  public function testExceptionInheritance(): void {
    $exception = new ApiConfigurationException('Test');
    $this->assertInstanceOf(\RuntimeException::class, $exception);
  }

  /**
   * Tests exception for missing API key scenario.
   */
  public function testMissingApiKeyScenario(): void {
    $exception = new ApiConfigurationException(
      'API key is not configured or invalid',
      'openai_chat_completions'
    );

    $this->assertStringContainsString('API key', $exception->getMessage());
    $this->assertEquals('openai_chat_completions', $exception->getPluginId());
  }

  /**
   * Tests exception for invalid endpoint configuration.
   */
  public function testInvalidEndpointScenario(): void {
    $exception = new ApiConfigurationException(
      'Invalid endpoint URL format',
      'anthropic_claude',
      400
    );

    $this->assertStringContainsString('endpoint', $exception->getMessage());
    $this->assertEquals('anthropic_claude', $exception->getPluginId());
    $this->assertEquals(400, $exception->getCode());
  }

}
