<?php

namespace Drupal\Tests\api_plugins\Unit\Exception;

use Drupal\api_plugins\Exception\ApiResponseException;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests for ApiResponseException.
 */
#[CoversClass(ApiResponseException::class)]
#[Group('api_plugins')]
class ApiResponseExceptionTest extends UnitTestCase {

  /**
   * Tests basic exception creation.
   */
  public function testBasicExceptionCreation(): void {
    $message = 'API request failed';
    $code = 500;
    $status_code = 404;
    $response_data = ['error' => 'Not found'];

    $exception = new ApiResponseException($message, $code, $status_code, $response_data);

    $this->assertEquals($message, $exception->getMessage());
    $this->assertEquals($code, $exception->getCode());
    $this->assertEquals($status_code, $exception->getStatusCode());
    $this->assertEquals($response_data, $exception->getResponseData());
  }

  /**
   * Tests exception without status code.
   */
  public function testExceptionWithoutStatusCode(): void {
    $exception = new ApiResponseException('Test error');

    $this->assertNull($exception->getStatusCode());
  }

  /**
   * Tests exception without response data.
   */
  public function testExceptionWithoutResponseData(): void {
    $exception = new ApiResponseException('Test error');

    $this->assertNull($exception->getResponseData());
  }

  /**
   * Tests exception with previous exception.
   */
  public function testExceptionWithPrevious(): void {
    $previous = new \RuntimeException('Previous error');
    $exception = new ApiResponseException('Current error', 0, NULL, NULL, $previous);

    $this->assertSame($previous, $exception->getPrevious());
  }

  /**
   * Tests that exception extends RuntimeException.
   */
  public function testExceptionInheritance(): void {
    $exception = new ApiResponseException('Test');
    $this->assertInstanceOf(\RuntimeException::class, $exception);
  }

  /**
   * Tests 401 Unauthorized scenario.
   */
  public function testUnauthorizedScenario(): void {
    $response_data = [
      'error' => [
        'message' => 'Invalid authentication',
        'type' => 'invalid_request_error',
      ],
    ];

    $exception = new ApiResponseException(
      'Authentication failed',
      0,
      401,
      $response_data
    );

    $this->assertEquals(401, $exception->getStatusCode());
    $this->assertEquals('Authentication failed', $exception->getMessage());
    $this->assertIsArray($exception->getResponseData());
    $this->assertArrayHasKey('error', $exception->getResponseData());
  }

  /**
   * Tests 500 Server Error scenario.
   */
  public function testServerErrorScenario(): void {
    $exception = new ApiResponseException(
      'Internal server error',
      0,
      500,
      ['error' => 'Server overloaded']
    );

    $this->assertEquals(500, $exception->getStatusCode());
    $this->assertStringContainsString('server error', $exception->getMessage());
  }

  /**
   * Tests rate limit exceeded scenario.
   */
  public function testRateLimitScenario(): void {
    $response_data = [
      'error' => [
        'message' => 'Rate limit exceeded',
        'type' => 'rate_limit_error',
      ],
    ];

    $exception = new ApiResponseException(
      'Rate limit exceeded',
      0,
      429,
      $response_data
    );

    $this->assertEquals(429, $exception->getStatusCode());
    $this->assertArrayHasKey('error', $exception->getResponseData());
  }

  /**
   * Tests parsing error scenario.
   */
  public function testParsingErrorScenario(): void {
    $invalid_json = 'This is not JSON';

    $exception = new ApiResponseException(
      'Failed to parse API response',
      0,
      NULL,
      $invalid_json
    );

    $this->assertEquals($invalid_json, $exception->getResponseData());
    $this->assertNull($exception->getStatusCode());
  }

}
