# API Status

Lightweight API status tracking module using Drupal's State API.

## Overview

Track the success and failure of any API calls using simple key-value storage:
- `api_status.success.{api_key}` = timestamp
- `api_status.failed.{api_key}` = timestamp

## Features

- **Simple service** - Easy to inject and use
- **State API storage** - No custom database tables
- **Dashboard** - View all tracked APIs at `/admin/reports/api-status`
- **No dependencies** - Uses only Drupal core

## Requirements

- Drupal 9, 10, or 11

## Installation

```bash
composer require drupal/api_status
drush en api_status
drush cr
```

## Usage

### Inject the service (recommended)

```php
use Drupal\api_status\ApiStatusServiceInterface;

class MyService {

  protected ApiStatusServiceInterface $tracker;

  public function __construct(ApiStatusServiceInterface $tracker) {
    $this->tracker = $tracker;
  }

  public function callExternalApi() {
    try {
      $response = $this->httpClient->request('POST', $url);
      $this->tracker->log('payment_api', 'success', $url);
      return $response;
    }
    catch (\Exception $e) {
      $this->tracker->log('payment_api', 'failed', $url);
      throw $e;
    }
  }

}
```

### Static usage

```php
$tracker = \Drupal::service('api_status.tracker');

// Log a successful API call
$tracker->log('stripe_payment', 'success', 'https://api.stripe.com/v1/charges');

// Log a failed API call
$tracker->log('sendgrid_email', 'failed', 'https://api.sendgrid.com/v3/mail/send');

// Get status of an API
$status = $tracker->getStatus('stripe_payment');
// Returns:
// [
//   'last_success' => 1733250000,
//   'last_failed' => NULL,
//   'current_status' => 'success',
//   'endpoint' => 'https://api.stripe.com/v1/charges'
// ]

// Get all tracked APIs
$all = $tracker->getAllStatuses();

// Clear single API
$tracker->clear('stripe_payment');

// Clear all
$tracker->clearAll();
```

## Dashboard

Visit `/admin/reports/api-status` to view all tracked APIs with their status.

## Storage Keys

The module uses these State API keys:
- `api_status.success.{api_key}` - Timestamp of last successful call
- `api_status.failed.{api_key}` - Timestamp of last failed call
- `api_status.endpoint.{api_key}` - The API endpoint/path
- `api_status.tracked` - Array of tracked API keys

## Permissions

- `access api status dashboard` - View the dashboard

## License

GPL-2.0-or-later

