<?php

namespace Drupal\api_status;

use Drupal\Core\State\StateInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Psr\Log\LoggerInterface;

/**
 * Service for API status tracking.
 */
class ApiStatusService implements ApiStatusServiceInterface {

  use StringTranslationTrait;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs an ApiStatusService object.
   *
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   */
  public function __construct(
    StateInterface $state,
    DateFormatterInterface $date_formatter,
    LoggerInterface $logger
  ) {
    $this->state = $state;
    $this->dateFormatter = $date_formatter;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public function log(string $api_key, string $status, ?string $endpoint = NULL): bool {
    if (!in_array($status, ['success', 'failed'])) {
      return FALSE;
    }

    $timestamp = time();

    // Store timestamp.
    $this->state->set("api_status.{$status}.{$api_key}", $timestamp);

    // Store endpoint path if provided.
    if ($endpoint) {
      $this->state->set("api_status.endpoint.{$api_key}", $endpoint);
    }

    // Track this API in the list of tracked APIs.
    $tracked_apis = $this->state->get('api_status.tracked', []);
    if (!in_array($api_key, $tracked_apis)) {
      $tracked_apis[] = $api_key;
      $this->state->set('api_status.tracked', $tracked_apis);
    }

    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getStatus(string $api_key): array {
    $last_success = $this->state->get("api_status.success.{$api_key}");
    $last_failed = $this->state->get("api_status.failed.{$api_key}");
    $endpoint = $this->state->get("api_status.endpoint.{$api_key}");

    $current_status = $this->determineStatus($last_success, $last_failed);

    return [
      'last_success' => $last_success,
      'last_failed' => $last_failed,
      'current_status' => $current_status,
      'endpoint' => $endpoint,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getAllStatuses(): array {
    $tracked_apis = $this->state->get('api_status.tracked', []);
    $statuses = [];

    foreach ($tracked_apis as $api_key) {
      $status = $this->getStatus($api_key);
      $statuses[$api_key] = [
        'key' => $api_key,
        'name' => $this->formatApiName($api_key),
        'endpoint' => $status['endpoint'],
        'current_status' => $status['current_status'],
        'last_success' => $status['last_success'],
        'last_failed' => $status['last_failed'],
      ];
    }

    return $statuses;
  }

  /**
   * {@inheritdoc}
   */
  public function clear(string $api_key): void {
    $this->state->delete("api_status.success.{$api_key}");
    $this->state->delete("api_status.failed.{$api_key}");
    $this->state->delete("api_status.endpoint.{$api_key}");

    // Remove from tracked list.
    $tracked_apis = $this->state->get('api_status.tracked', []);
    $tracked_apis = array_diff($tracked_apis, [$api_key]);
    $this->state->set('api_status.tracked', array_values($tracked_apis));
  }

  /**
   * {@inheritdoc}
   */
  public function clearAll(): void {
    $tracked_apis = $this->state->get('api_status.tracked', []);

    foreach ($tracked_apis as $api_key) {
      $this->state->delete("api_status.success.{$api_key}");
      $this->state->delete("api_status.failed.{$api_key}");
      $this->state->delete("api_status.endpoint.{$api_key}");
    }

    $this->state->delete('api_status.tracked');
  }

  /**
   * Formats timestamp as "time ago".
   *
   * @param int|null $timestamp
   *   Unix timestamp or NULL.
   *
   * @return string
   *   Formatted string like "5 minutes ago" or "Never".
   */
  public function formatTimeAgo(?int $timestamp): string {
    if (!$timestamp) {
      return $this->t('Never')->render();
    }

    return $this->dateFormatter->formatTimeDiffSince($timestamp);
  }

  /**
   * Determines the current status based on timestamps.
   *
   * @param int|null $last_success
   *   Last success timestamp.
   * @param int|null $last_failed
   *   Last failed timestamp.
   *
   * @return string
   *   The status: 'success', 'failed', or 'unknown'.
   */
  protected function determineStatus(?int $last_success, ?int $last_failed): string {
    if ($last_success && $last_failed) {
      return ($last_success > $last_failed) ? 'success' : 'failed';
    }
    elseif ($last_success) {
      return 'success';
    }
    elseif ($last_failed) {
      return 'failed';
    }

    return 'unknown';
  }

  /**
   * Formats an API key into a readable name.
   *
   * @param string $api_key
   *   The API key.
   *
   * @return string
   *   Formatted name.
   */
  protected function formatApiName(string $api_key): string {
    return ucwords(str_replace('_', ' ', $api_key));
  }

}

