<?php

namespace Drupal\api_status\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\api_status\ApiStatusServiceInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for the API Status dashboard.
 */
class DashboardController extends ControllerBase {

  /**
   * The API status service.
   *
   * @var \Drupal\api_status\ApiStatusServiceInterface
   */
  protected $apiStatus;

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * Constructs a DashboardController object.
   *
   * @param \Drupal\api_status\ApiStatusServiceInterface $api_status
   *   The API status service.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   */
  public function __construct(ApiStatusServiceInterface $api_status, DateFormatterInterface $date_formatter) {
    $this->apiStatus = $api_status;
    $this->dateFormatter = $date_formatter;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('api_status.tracker'),
      $container->get('date.formatter')
    );
  }

  /**
   * Displays the API Status dashboard.
   */
  public function dashboard() {
    $build = [];

    // Tracked APIs Section.
    $build['tracked_section'] = [
      '#type' => 'details',
      '#title' => $this->t('Tracked APIs'),
      '#open' => TRUE,
    ];

    $build['tracked_section']['description'] = [
      '#markup' => '<p>' . $this->t('APIs tracked via the API Status service.') . '</p>',
    ];

    $build['tracked_section']['table'] = $this->buildTrackedApisTable();

    // Usage Instructions.
    $build['usage_section'] = [
      '#type' => 'details',
      '#title' => $this->t('How to Track APIs'),
      '#open' => FALSE,
    ];

    $build['usage_section']['code'] = [
      '#markup' => '<pre><code>' . htmlspecialchars("// Get the service
\$tracker = \\Drupal::service('api_status.tracker');

// Log a successful API call
\$tracker->log('stripe_payment', 'success', 'https://api.stripe.com/v1/charges');

// Log a failed API call
\$tracker->log('sendgrid_email', 'failed', 'https://api.sendgrid.com/v3/mail/send');

// Get status of an API
\$status = \$tracker->getStatus('stripe_payment');

// Get all tracked APIs
\$all = \$tracker->getAllStatuses();

// Clear single API
\$tracker->clear('stripe_payment');

// Clear all
\$tracker->clearAll();") . '</code></pre>',
    ];

    $build['#cache']['max-age'] = 0;

    return $build;
  }

  /**
   * Builds the tracked APIs table.
   */
  protected function buildTrackedApisTable() {
    $tracked_apis = $this->apiStatus->getAllStatuses();

    if (empty($tracked_apis)) {
      return [
        '#markup' => '<p>' . $this->t('No APIs tracked yet.') . '</p>',
      ];
    }

    $header = [
      ['data' => $this->t('API Name')],
      ['data' => $this->t('Key')],
      ['data' => $this->t('Endpoint')],
      ['data' => $this->t('Status')],
      ['data' => $this->t('Last Success')],
      ['data' => $this->t('Last Failed')],
    ];

    $rows = [];
    foreach ($tracked_apis as $api) {
      $rows[] = [
        'data' => [
          $api['name'],
          ['data' => ['#markup' => '<code>' . $api['key'] . '</code>']],
          ['data' => ['#markup' => $api['endpoint'] ? '<code>' . $this->truncate($api['endpoint'], 40) . '</code>' : '-']],
          ['data' => ['#markup' => $this->formatStatus($api['current_status'])]],
          ['data' => ['#markup' => $this->formatTime($api['last_success'])]],
          ['data' => ['#markup' => $this->formatTime($api['last_failed'])]],
        ],
      ];
    }

    return [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No APIs tracked yet.'),
    ];
  }

  /**
   * Formats status.
   */
  protected function formatStatus($status) {
    switch ($status) {
      case 'success':
        return $this->t('Success');

      case 'failed':
        return $this->t('Failed');

      default:
        return $this->t('Unknown');
    }
  }

  /**
   * Formats a timestamp.
   */
  protected function formatTime($timestamp) {
    if (!$timestamp) {
      return $this->t('Never');
    }

    return $this->dateFormatter->formatTimeDiffSince($timestamp);
  }

  /**
   * Truncates a string.
   */
  protected function truncate($string, $length = 50) {
    if (strlen($string) <= $length) {
      return htmlspecialchars($string);
    }
    return htmlspecialchars(substr($string, 0, $length - 3)) . '...';
  }

}

